<?php
session_start();
include($_SERVER['DOCUMENT_ROOT'] . '/config.php');

// Verify admin is logged in
if (!isset($_SESSION['admin_id']) || empty($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}

// Initialize variables
$plans = [];
$investments = [];
$users = [];
$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Plan operations
    if (isset($_POST['action'])) {
        switch($_POST['action']) {
            case 'create_plan':
                createPlan($conn);
                break;
            case 'update_plan':
                updatePlan($conn);
                break;
            case 'delete_plan':
                deletePlan($conn);
                break;
            case 'update_investment':
                updateInvestment($conn);
                break;
        }
    }
}

// Functions for CRUD operations
function createPlan($conn) {
    global $message, $error;
    
    try {
        $name = $_POST['name'];
        $amount = $_POST['amount'];
        $duration = $_POST['duration'];
        $description = $_POST['description'];
        $returnRate = $_POST['return_rate'];
        $currency = $_POST['currency'];
        
        $sql = "INSERT INTO plans (name, amount, duration, description, return_rate, currency) 
                VALUES (:name, :amount, :duration, :description, :return_rate, :currency)";
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':duration', $duration);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':return_rate', $returnRate);
        $stmt->bindParam(':currency', $currency);
        
        if ($stmt->execute()) {
            $message = "Plan created successfully";
        } else {
            $error = "Failed to create plan";
        }
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
}

function updatePlan($conn) {
    global $message, $error;
    
    try {
        $id = $_POST['plan_id'];
        $name = $_POST['name'];
        $amount = $_POST['amount'];
        $duration = $_POST['duration'];
        $description = $_POST['description'];
        $returnRate = $_POST['return_rate'];
        $currency = $_POST['currency'];
        
        $sql = "UPDATE plans 
                SET name = :name, amount = :amount, duration = :duration, 
                    description = :description, return_rate = :return_rate, currency = :currency 
                WHERE id = :id";
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':duration', $duration);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':return_rate', $returnRate);
        $stmt->bindParam(':currency', $currency);
        
        if ($stmt->execute()) {
            $message = "Plan updated successfully";
        } else {
            $error = "Failed to update plan";
        }
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
}

function deletePlan($conn) {
    global $message, $error;
    
    try {
        $id = $_POST['plan_id'];
        
        // Check if plan is being used by any investments
        $checkSql = "SELECT COUNT(*) FROM investments WHERE plan_id = :id AND status IN ('PENDING', 'ACTIVE')";
        $checkStmt = $conn->prepare($checkSql);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->execute();
        
        if ($checkStmt->fetchColumn() > 0) {
            $error = "Cannot delete plan that has active investments";
            return;
        }
        
        $sql = "DELETE FROM plans WHERE id = :id";
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            $message = "Plan deleted successfully";
        } else {
            $error = "Failed to delete plan";
        }
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
}

function updateInvestment($conn) {
    global $message, $error;
    
    try {
        $id = $_POST['investment_id'];
        $status = $_POST['status'];
        $userId = $_POST['user_id'];
        $amount = $_POST['amount'];
        $currency = $_POST['currency'];
        $oldStatus = $_POST['old_status'];
        
        $conn->beginTransaction();
        
        // Update investment status
        $sql = "UPDATE investments SET status = :status WHERE id = :id";
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':status', $status);
        $stmt->execute();
        
        // Handle status transitions
        if ($status == 'COMPLETED' && $oldStatus != 'COMPLETED') {
            // When investment completes, move money from investment to profit
            $updateSql = "UPDATE users SET investment_balance = investment_balance - :amount, profit_balance = profit_balance + :amount WHERE id = :user_id";
            $updateStmt = $conn->prepare($updateSql);
            $updateStmt->bindParam(':amount', $amount);
            $updateStmt->bindParam(':user_id', $userId);
            $updateStmt->execute();
        } else if ($status == 'CANCELLED' && $oldStatus != 'CANCELLED') {
            // When investment is cancelled, return money to appropriate balance
            $balanceField = strtolower($currency) . '_balance';
            $updateSql = "UPDATE users SET {$balanceField} = {$balanceField} + :amount, investment_balance = investment_balance - :amount WHERE id = :user_id";
            $updateStmt = $conn->prepare($updateSql);
            $updateStmt->bindParam(':amount', $amount);
            $updateStmt->bindParam(':user_id', $userId);
            $updateStmt->execute();
        } else if ($oldStatus == 'PENDING' && $status == 'ACTIVE') {
            // Just a status change, no balance updates needed
        }
        
        $conn->commit();
        $message = "Investment status updated successfully";
        
    } catch (Exception $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        $error = "Error: " . $e->getMessage();
    }
}

// Fetch all plans
try {
    $plansSql = "SELECT * FROM plans ORDER BY id DESC";
    $plansStmt = $conn->prepare($plansSql);
    $plansStmt->execute();
    $plans = $plansStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error = "Error fetching plans: " . $e->getMessage();
}

// Fetch all investments
try {
    $investmentsSql = "SELECT i.*, u.username, u.email FROM investments i 
                       JOIN users u ON i.user_id = u.id 
                       ORDER BY i.start_date DESC";
    $investmentsStmt = $conn->prepare($investmentsSql);
    $investmentsStmt->execute();
    $investments = $investmentsStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error = "Error fetching investments: " . $e->getMessage();
}

// Fetch all users
try {
    $usersSql = "SELECT id, username, email, balance, usdc_balance, btc_balance, investment_balance, profit_balance 
                FROM users ORDER BY id";
    $usersStmt = $conn->prepare($usersSql);
    $usersStmt->execute();
    $users = $usersStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error = "Error fetching users: " . $e->getMessage();
}

// Close the connection
$conn = null;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        :root {
            --sidebar-bg: #14213d;
            --sidebar-text: #a8b2d1;
            --sidebar-hover: #1a2c5b;
            --active-link: #2a4494;
            --primary-text: #333;
            --secondary-text: #777;
            --card-bg: #fff;
            --main-bg: #f0f2f5;
            --deposit-color: #3498db;
            --withdraw-color: #e74c3c;
            --button-color: #0d6efd;
            --hover-color: #0b5ed7;
        }

        body {
            display: flex;
            background-color: var(--main-bg);
            min-height: 100vh;
        }

        /* Mobile nav toggle */
        .mobile-nav-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            width: 40px;
            height: 40px;
            border-radius: 4px;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            border: none;
            font-size: 20px;
        }

        /* Overlay for mobile */
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 900;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            height: 100vh;
            position: fixed;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        .sidebar-header {
            padding: 20px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-menu {
            margin-top: 20px;
        }

        .section-title {
            padding: 10px 20px;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #8a94a7;
            margin-top: 10px;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            text-decoration: none;
            color: var(--sidebar-text);
            transition: all 0.3s ease;
        }

        .menu-item:hover {
            background-color: var(--sidebar-hover);
        }

        .menu-item.active {
            background-color: var(--active-link);
            color: white;
        }

        .menu-item i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 20px;
            width: calc(100% - 250px);
            transition: all 0.3s ease;
        }

        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .page-icon {
            font-size: 20px;
            color: var(--primary-text);
        }

        .user-info {
            display: flex;
            align-items: center;
        }

        .user-icon {
            width: 32px;
            height: 32px;
            background-color: var(--button-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-right: 10px;
        }

        .user-role {
            display: flex;
            flex-direction: column;
        }

        .role-name {
            font-size: 12px;
            color: var(--secondary-text);
        }

        /* Overview Section */
        .overview-section {
            margin-bottom: 30px;
        }

        .overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .overview-title {
            font-size: 24px;
            font-weight: 500;
            color: var(--primary-text);
        }

        .overview-description {
            font-size: 14px;
            color: var(--secondary-text);
            margin-top: 5px;
        }

        .view-users-btn {
            padding: 8px 16px;
            background-color: var(--button-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .view-users-btn:hover {
            background-color: var(--hover-color);
        }

        .view-users-btn i {
            margin-right: 6px;
        }

        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background-color: var(--card-bg);
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 20px;
            position: relative;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .card-title {
            font-size: 16px;
            color: var(--primary-text);
        }

        .card-icon {
            font-size: 14px;
            color: var(--secondary-text);
            cursor: pointer;
        }

        .card-amount {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .deposit-bar {
            height: 4px;
            width: 100%;
            background-color: var(--deposit-color);
            border-radius: 2px;
        }

        .withdraw-bar {
            height: 4px;
            width: 100%;
            background-color: var(--withdraw-color);
            border-radius: 2px;
        }

        /* Stats Card */
        .stats-card {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }

        .stat-item {
            margin-bottom: 15px;
        }

        .stat-label {
            display: flex;
            align-items: center;
            margin-bottom: 5px;
            font-size: 14px;
            color: var(--secondary-text);
        }

        .stat-label-circle {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .deposit-circle {
            background-color: #f39c12;
        }

        .withdraw-circle {
            background-color: #e74c3c;
        }

        .transaction-circle {
            background-color: #9b59b6;
        }

        .users-circle {
            background-color: #3498db;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 500;
        }

        /* Footer */
        .footer {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 40px;
            padding-top: 10px;
            border-top: 1px solid #eee;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .sidebar {
                width: 80px;
            }
            .sidebar-header {
                padding: 15px 10px;
                text-align: center;
                font-size: 14px;
            }
            .menu-item span, .section-title {
                display: none;
            }
            .menu-item {
                justify-content: center;
                padding: 15px 0;
            }
            .menu-item i {
                margin-right: 0;
                font-size: 18px;
            }
            .main-content {
                margin-left: 80px;
                width: calc(100% - 80px);
            }
        }

        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }
            .stats-card {
                grid-template-columns: 1fr;
            }
            .mobile-nav-toggle {
                display: flex;
            }
            .sidebar {
                transform: translateX(-100%);
                width: 250px;
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .menu-item span, .section-title {
                display: block;
            }
            .menu-item {
                justify-content: flex-start;
                padding: 12px 20px;
            }
            .menu-item i {
                margin-right: 10px;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding-top: 60px;
            }
            .overlay.active {
                display: block;
            }
        }

        @media (max-width: 576px) {
            .overview-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .view-users-btn {
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Mobile menu toggle -->
    <button class="mobile-nav-toggle" id="navToggle">
        <i class="fas fa-bars"></i>
    </button>

        <!-- Overlay for closing the sidebar on mobile -->
    <div class="overlay" id="overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header">ADMIN DASHBOARD</div>
        <div class="sidebar-menu">
            <a href="/admin/admin_dashboard.php" class="menu-item active">
                <i class="fas fa-th-large"></i>
                <span>Dashboard</span>
            </a>
            <a href="/admin/coin-deposit-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-down"></i>
                <span>Deposit Coin</span>
            </a>
             <a href="/admin/deposit_requests.php" class="menu-item">
                <i class="fas fa-money-check-alt"></i>
                <span>Deposit Check</span>
            </a>
            <a href="/admin/transfer-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-up"></i>
                <span>Withdraw</span>
            </a>
            <a href="/admin/manage-payment-method.php" class="menu-item">
                <i class="fas fa-exchange-alt"></i>
                <span>Manage Payment Method</span>
            </a>
            <a href="/admin/admin_staking.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking</span>
            </a>
            <a href="/admin/admin_staking_transactions.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking Transations</span>
            </a>
            <a href="/admin/admin_stocks.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks</span>
            </a>
            <a href="/admin/admin_stock_transactions.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks Transations</span>
            </a>
            <a href="/admin/admin_bots.php" class="menu-item">
                <i class="fas fa-robot"></i>
                <span>Manage Bot</span>
            </a>
            <a href="/admin/investment.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Investment</span>
            </a>
            
            <a href="/admin/crypto_orders.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Crypto Order</span>
            </a>
            
            <a href="/admin/update_balance.php" class="menu-item">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Update Balances</span>
            </a>
            <a href="/admin/kyc-request.php" class="menu-item">
                <i class="fas fa-id-card"></i>
                <span>KYC</span>
            </a>
            <a href="/admin/admin_send_notification.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Manage Notifications</span>
            </a>

            <div class="section-title">USER MANAGEMENT</div>
            <a href="/admin/manage_users.php" class="menu-item">
                <i class="fas fa-users"></i>
                <span>Manage Users</span>
            </a>
            <div class="section-title">Approve SignUp</div>
            <a href="/admin/admin_approval.php" class="menu-item">
                <i class="fas fa-user-check"></i>
                <span>Approve SignUp</span>
            </a>
            <div class="section-title">Change Password</div>
            <a href="/admin/change_password.php" class="menu-item">
                <i class="fas fa-unlock-alt"></i>
                <span>Change Password</span>
            </a>

            <div class="section-title">MANAGEMENT</div>
            <a href="/admin/logout.php" class="menu-item">
                <i class="fas fa-sign-out"></i>
                <span>Sign Out</span>
            </a>
        </div>
    </div>


    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="top-bar">
            <div class="page-icon">
                <i class="fas fa-clipboard"></i>
            </div>
            <div class="user-info">
                <div class="user-icon">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-role">
                    <div>Admin</div>
                    <div class="role-name">Administrator <i class="fas fa-chevron-down"></i></div>
                </div>
            </div>
        </div>

        <!-- Overview Section -->
        <div class="overview-section">
            <div class="overview-header">
                <div>
                    <h2 class="overview-title">Overview</h2>
                    <p class="overview-description">Here is an insight of what's going on.</p>
                </div>
                <button class="view-users-btn">
                    <i class="fas fa-users"></i> View All Users
                </button>
            </div>

    <style>
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        .tabs {
            display: flex;
            margin-bottom: 20px;
            border-bottom: 1px solid #ddd;
        }
        .tab {
            padding: 10px 20px;
            cursor: pointer;
            border: 1px solid transparent;
            border-bottom: none;
        }
        .tab.active {
            background-color: #fff;
            border-color: #ddd;
            border-radius: 5px 5px 0 0;
        }
        .tab-content {
            display: none;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .tab-content.active {
            display: block;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #f5f5f5;
        }
        form {
            margin-bottom: 20px;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input[type="text"],
        input[type="number"],
        select,
        textarea {
            width: 100%;
            padding: 8px;
            box-sizing: border-box;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .button {
            padding: 10px 15px;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        .button.edit {
            background-color: #2196F3;
        }
        .button.delete {
            background-color: #f44336;
        }
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
        }
        .alert-success {
            background-color: #dff0d8;
            color: #3c763d;
        }
        .alert-danger {
            background-color: #f2dede;
            color: #a94442;
        }
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }
        .modal-content {
            background-color: #fff;
            margin: 10% auto;
            padding: 20px;
            width: 60%;
            border-radius: 5px;
        }
        .close {
            float: right;
            cursor: pointer;
            font-size: 20px;
        }
          /* Enhanced scrolling behavior */
        html {
            height: 100%;
            overflow: hidden;
        }
        
        body {
            height: 100%;
            overflow: auto;
            -webkit-overflow-scrolling: touch; /* Smooth scrolling on iOS */
        }
        
        /* Improved modal scrolling */
        .modal {
            overflow-y: auto;
            -webkit-overflow-scrolling: touch;
            padding: 20px 0;
        }
        
        .modal-content {
            margin: 20px auto;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        /* Better mobile table display */
        @media (max-width: 768px) {
            table {
                display: block;
                width: 100%;
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            th, td {
                min-width: 120px; /* Prevent columns from becoming too narrow */
            }
            
            .tab {
                padding: 10px 15px;
                font-size: 14px;
            }
            
            .button {
                padding: 8px 12px;
                font-size: 14px;
            }
        }
        
        /* Loading indicator */
        .loading {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            z-index: 9999;
            justify-content: center;
            align-items: center;
        }
        
        .loading-spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #3498db;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 2s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Back to top button */
        .back-to-top {
            display: none;
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 99;
            border: none;
            outline: none;
            background-color: #4CAF50;
            color: white;
            cursor: pointer;
            padding: 12px;
            border-radius: 50%;
            font-size: 18px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
        }
        
        .back-to-top:hover {
            background-color: #45a049;
        }
    </style>
    <!-- Loading indicator -->
    <div class="loading" id="loadingIndicator">
        <div class="loading-spinner"></div>
    </div>
    
    <!-- Back to top button -->
    <button onclick="scrollToTop()" class="back-to-top" id="backToTopBtn" title="Go to top">↑</button>
    <div class="container">

        <?php if ($message): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="tabs">
            <div class="tab active" data-tab="plans">Plans</div>
            <div class="tab" data-tab="investments">Investments</div>
            <div class="tab" data-tab="users">Users</div>
        </div>
        
        <!-- Plans Tab -->
        <div id="plans" class="tab-content active">
            <h2>Investment Plans</h2>
            
            <button class="button" onclick="toggleModal('createPlanModal')">Create New Plan</button>
            
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Amount</th>
                        <th>Currency</th>
                        <th>Duration (Days)</th>
                        <th>Return Rate (%)</th>
                        <th>Description</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($plans as $plan): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($plan['id']); ?></td>
                            <td><?php echo htmlspecialchars($plan['name']); ?></td>
                            <td><?php echo htmlspecialchars($plan['amount']); ?></td>
                            <td><?php echo htmlspecialchars($plan['currency']); ?></td>
                            <td><?php echo htmlspecialchars($plan['duration']); ?></td>
                            <td><?php echo htmlspecialchars($plan['return_rate']); ?></td>
                            <td><?php echo htmlspecialchars($plan['description']); ?></td>
                            <td>
                                <button class="button edit" onclick="editPlan(<?php echo htmlspecialchars(json_encode($plan)); ?>)">Edit</button>
                                <button class="button delete" onclick="deletePlanConfirm(<?php echo $plan['id']; ?>)">Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Investments Tab -->
        <div id="investments" class="tab-content">
            <h2>Investment Orders</h2>
            
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>User</th>
                        <th>Plan</th>
                        <th>Amount</th>
                        <th>Currency</th>
                        <th>Status</th>
                        <th>Start Date</th>
                        <th>End Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($investments as $investment): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($investment['id']); ?></td>
                            <td><?php echo htmlspecialchars($investment['username']) . ' (' . htmlspecialchars($investment['email']) . ')'; ?></td>
                            <td><?php echo htmlspecialchars($investment['plan_name']); ?></td>
                            <td><?php echo htmlspecialchars($investment['amount']); ?></td>
                            <td><?php echo htmlspecialchars($investment['currency']); ?></td>
                            <td><?php echo htmlspecialchars($investment['status']); ?></td>
                            <td><?php echo htmlspecialchars($investment['start_date']); ?></td>
                            <td><?php echo htmlspecialchars($investment['end_date']); ?></td>
                            <td>
                                <button class="button edit" onclick="editInvestment(<?php echo htmlspecialchars(json_encode($investment)); ?>)">Update Status</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Users Tab -->
        <div id="users" class="tab-content">
            <h2>User Management</h2>
            
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Email</th>
                        <th>USD Balance</th>
                        <th>USDC Balance</th>
                        <th>BTC Balance</th>
                        <th>Investment Balance</th>
                        <th>Profit Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($user['id']); ?></td>
                            <td><?php echo htmlspecialchars($user['username']); ?></td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td>$<?php echo number_format($user['balance'], 2); ?></td>
                            <td><?php echo number_format($user['usdc_balance'], 6); ?> USDC</td>
                            <td><?php echo number_format($user['btc_balance'], 8); ?> BTC</td>
                            <td>$<?php echo number_format($user['investment_balance'], 2); ?></td>
                            <td>$<?php echo number_format($user['profit_balance'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Create Plan Modal -->
        <div id="createPlanModal" class="modal">
            <div class="modal-content">
                <span class="close" onclick="toggleModal('createPlanModal')">&times;</span>
                <h2>Create New Plan</h2>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="create_plan">
                    
                    <div class="form-group">
                        <label for="name">Plan Name:</label>
                        <input type="text" id="name" name="name" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="amount">Amount:</label>
                        <input type="number" id="amount" name="amount" step="0.00000001" min="0" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="currency">Currency:</label>
                        <select id="currency" name="currency" required>
                            <option value="USD">USD</option>
                            <option value="USDC">USDC</option>
                            <option value="BTC">BTC</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="duration">Duration (days):</label>
                        <input type="number" id="duration" name="duration" min="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="return_rate">Return Rate (%):</label>
                        <input type="number" id="return_rate" name="return_rate" step="0.01" min="0" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="description">Description:</label>
                        <textarea id="description" name="description" rows="3" required></textarea>
                    </div>
                    
                    <button type="submit" class="button">Create Plan</button>
                </form>
            </div>
        </div>
        
        <!-- Edit Plan Modal -->
        <div id="editPlanModal" class="modal">
            <div class="modal-content">
                <span class="close" onclick="toggleModal('editPlanModal')">&times;</span>
                <h2>Edit Plan</h2>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="update_plan">
                    <input type="hidden" id="edit_plan_id" name="plan_id">
                    
                    <div class="form-group">
                        <label for="edit_name">Plan Name:</label>
                        <input type="text" id="edit_name" name="name" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_amount">Amount:</label>
                        <input type="number" id="edit_amount" name="amount" step="0.00000001" min="0" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_currency">Currency:</label>
                        <select id="edit_currency" name="currency" required>
                            <option value="USD">USD</option>
                            <option value="USDC">USDC</option>
                            <option value="BTC">BTC</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_duration">Duration (days):</label>
                        <input type="number" id="edit_duration" name="duration" min="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_return_rate">Return Rate (%):</label>
                        <input type="number" id="edit_return_rate" name="return_rate" step="0.01" min="0" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_description">Description:</label>
                        <textarea id="edit_description" name="description" rows="3" required></textarea>
                    </div>
                    
                    <button type="submit" class="button">Update Plan</button>
                </form>
            </div>
        </div>
        
        <!-- Delete Plan Modal -->
        <div id="deletePlanModal" class="modal">
            <div class="modal-content">
                <span class="close" onclick="toggleModal('deletePlanModal')">&times;</span>
                <h2>Confirm Delete</h2>
                <p>Are you sure you want to delete this plan? This action cannot be undone.</p>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="delete_plan">
                    <input type="hidden" id="delete_plan_id" name="plan_id">
                    <button type="submit" class="button delete">Delete</button>
                    <button type="button" class="button" onclick="toggleModal('deletePlanModal')">Cancel</button>
                </form>
            </div>
        </div>
        
        <!-- Edit Investment Modal -->
        <div id="editInvestmentModal" class="modal">
            <div class="modal-content">
                <span class="close" onclick="toggleModal('editInvestmentModal')">&times;</span>
                <h2>Update Investment Status</h2>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="update_investment">
                    <input type="hidden" id="edit_investment_id" name="investment_id">
                    <input type="hidden" id="edit_investment_user_id" name="user_id">
                    <input type="hidden" id="edit_investment_amount" name="amount">
                    <input type="hidden" id="edit_investment_currency" name="currency">
                    <input type="hidden" id="edit_investment_old_status" name="old_status">
                    
                    <div class="form-group">
                        <label for="edit_investment_status">Status:</label>
                        <select id="edit_investment_status" name="status" required>
                            <option value="PENDING">PENDING</option>
                            <option value="ACTIVE">ACTIVE</option>
                            <option value="COMPLETED">COMPLETED</option>
                            <option value="CANCELLED">CANCELLED</option>
                        </select>
                    </div>
                    
                    <p><strong>Note:</strong> Status changes affect user balances:</p>
                    <ul>
                        <li><strong>PENDING → ACTIVE:</strong> No balance change, just status update</li>
                        <li><strong>Any → COMPLETED:</strong> Moves money from investment to profit balance</li>
                        <li><strong>Any → CANCELLED:</strong> Returns money to original currency balance</li>
                    </ul>
                    
                    <button type="submit" class="button">Update Status</button>
                </form>
            </div>
        </div>
        
        <script>
         // Get references to elements
        const navToggle = document.getElementById('navToggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');
        const mainContent = document.getElementById('mainContent');

        // Toggle sidebar
        navToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        });

        // Close sidebar when clicking on overlay
        overlay.addEventListener('click', function() {
            sidebar.classList.remove('active');
            overlay.classList.remove('active');
        });

        // Close sidebar when clicking on main content (for mobile)
        mainContent.addEventListener('click', function() {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Close sidebar when clicking menu items (for mobile)
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                }
            });
        });
            // Tab switching
            document.querySelectorAll('.tab').forEach(tab => {
                tab.addEventListener('click', function() {
                    // Remove active class from all tabs and content
                    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                    document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                    
                    // Add active class to current tab and content
                    this.classList.add('active');
                    document.getElementById(this.dataset.tab).classList.add('active');
                });
            });
            
            // Toggle modal visibility
            function toggleModal(modalId) {
                const modal = document.getElementById(modalId);
                modal.style.display = modal.style.display === 'block' ? 'none' : 'block';
            }
            
            // When user clicks anywhere outside of the modal, close it
            window.onclick = function(event) {
                if (event.target.classList.contains('modal')) {
                    event.target.style.display = 'none';
                }
            }
            
            // Edit plan
            function editPlan(plan) {
                document.getElementById('edit_plan_id').value = plan.id;
                document.getElementById('edit_name').value = plan.name;
                document.getElementById('edit_amount').value = plan.amount;
                document.getElementById('edit_currency').value = plan.currency;
                document.getElementById('edit_duration').value = plan.duration;
                document.getElementById('edit_return_rate').value = plan.return_rate;
                document.getElementById('edit_description').value = plan.description;
                
                toggleModal('editPlanModal');
            }
            
            // Delete plan confirmation
            function deletePlanConfirm(planId) {
                document.getElementById('delete_plan_id').value = planId;
                toggleModal('deletePlanModal');
            }
            
            // Edit investment
            function editInvestment(investment) {
                document.getElementById('edit_investment_id').value = investment.id;
                document.getElementById('edit_investment_user_id').value = investment.user_id;
                document.getElementById('edit_investment_amount').value = investment.amount;
                document.getElementById('edit_investment_currency').value = investment.currency;
                document.getElementById('edit_investment_status').value = investment.status;
                document.getElementById('edit_investment_old_status').value = investment.status;
                
                toggleModal('editInvestmentModal');
            }
              // Enhanced scrolling behavior
        document.addEventListener('DOMContentLoaded', function() {
            // Show/hide back to top button
            window.onscroll = function() {
                const backToTopBtn = document.getElementById('backToTopBtn');
                if (document.body.scrollTop > 20 || document.documentElement.scrollTop > 20) {
                    backToTopBtn.style.display = "block";
                } else {
                    backToTopBtn.style.display = "none";
                }
            };
            
            // Smooth scrolling for anchor links
            document.querySelectorAll('a[href^="#"]').forEach(anchor => {
                anchor.addEventListener('click', function(e) {
                    e.preventDefault();
                    document.querySelector(this.getAttribute('href')).scrollIntoView({
                        behavior: 'smooth'
                    });
                });
            });
            
            // Prevent body scroll when modal is open
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                modal.addEventListener('show', function() {
                    document.body.style.overflow = 'hidden';
                });
                modal.addEventListener('hide', function() {
                    document.body.style.overflow = '';
                });
            });
            
            // Improved tab switching with smooth transitions
            document.querySelectorAll('.tab').forEach(tab => {
                tab.addEventListener('click', function() {
                    showLoading();
                    
                    // Use setTimeout to allow UI to update before heavy DOM operations
                    setTimeout(() => {
                        document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                        document.querySelectorAll('.tab-content').forEach(c => {
                            c.style.opacity = '0';
                            c.classList.remove('active');
                        });
                        
                        this.classList.add('active');
                        const activeContent = document.getElementById(this.dataset.tab);
                        activeContent.classList.add('active');
                        
                        // Smooth fade-in effect
                        setTimeout(() => {
                            activeContent.style.opacity = '1';
                            hideLoading();
                            
                            // Scroll to top of the tab content
                            activeContent.scrollTo({top: 0, behavior: 'smooth'});
                        }, 50);
                    }, 10);
                });
            });
            
            // Initialize tab content opacity
            document.querySelectorAll('.tab-content').forEach(content => {
                content.style.transition = 'opacity 0.3s ease';
                if (content.classList.contains('active')) {
                    content.style.opacity = '1';
                } else {
                    content.style.opacity = '0';
                }
            });
        });
        
        // Scroll to top function
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Loading indicator functions
        function showLoading() {
            document.getElementById('loadingIndicator').style.display = 'flex';
        }
        
        function hideLoading() {
            document.getElementById('loadingIndicator').style.display = 'none';
        }
        
        // Enhanced modal handling
        function toggleModal(modalId) {
            const modal = document.getElementById(modalId);
            if (modal.style.display === 'block') {
                modal.style.display = 'none';
                modal.dispatchEvent(new Event('hide'));
            } else {
                // Close any other open modals first
                document.querySelectorAll('.modal').forEach(m => {
                    if (m.id !== modalId && m.style.display === 'block') {
                        m.style.display = 'none';
                        m.dispatchEvent(new Event('hide'));
                    }
                });
                
                modal.style.display = 'block';
                modal.dispatchEvent(new Event('show'));
                
                // Scroll to top of modal
                setTimeout(() => {
                    modal.scrollTo({top: 0, behavior: 'smooth'});
                }, 10);
            }
        }
        
        // Improved table scrolling on mobile
        function setupTableScrolling() {
            const tables = document.querySelectorAll('table');
            tables.forEach(table => {
                // Only apply touch scrolling for mobile devices
                if ('ontouchstart' in window) {
                    table.addEventListener('touchstart', function(e) {
                        this.style.overflowX = 'auto';
                    }, {passive: true});
                    
                    table.addEventListener('touchend', function(e) {
                        this.style.overflowX = 'hidden';
                    }, {passive: true});
                }
            });
        }
        
        // Initialize table scrolling
        setupTableScrolling();
        </script>
    </div>
</body>
</html>