<?php
// admin_stocks.php
include($_SERVER['DOCUMENT_ROOT'] . '/config.php');
// Verify admin is logged in
if (!isset($_SESSION['admin_id']) || empty($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}
// Enable error logging but disable display
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/error.log');

// Initialize variables
$message = '';
$messageType = '';

// Handle form submission via AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    $response = ['success' => false, 'message' => ''];
    
    try {
        if ($_POST['action'] === 'create' || $_POST['action'] === 'update') {
            // Validate inputs
            $id = isset($_POST['id']) && is_numeric($_POST['id']) ? $_POST['id'] : null;
            $name = trim($_POST['name'] ?? '');
            $symbol = trim($_POST['symbol'] ?? '');
            $minimum = isset($_POST['minimum']) && is_numeric($_POST['minimum']) ? (float)$_POST['minimum'] : 0;
            $maximum = isset($_POST['maximum']) && is_numeric($_POST['maximum']) ? (float)$_POST['maximum'] : 0;
            $cycle = trim($_POST['cycle'] ?? '');
            $returnRate = isset($_POST['return_rate']) && is_numeric($_POST['return_rate']) ? (float)$_POST['return_rate'] : 0;
            $status = in_array($_POST['status'] ?? '', ['active', 'inactive']) ? $_POST['status'] : 'inactive';

            // Validation
            if (empty($name) || empty($symbol) || empty($cycle)) {
                throw new Exception("All fields are required.");
            }

            if ($minimum < 0 || $maximum < 0) {
                throw new Exception("Minimum and maximum amounts must be positive.");
            }

            if ($minimum > $maximum) {
                throw new Exception("Minimum amount cannot be greater than maximum amount.");
            }

            if ($returnRate < -100 || $returnRate > 1000) {
                throw new Exception("Return rate must be between -100% and 1000%");
            }

            // Update or Insert
            if ($id && $_POST['action'] === 'update') {
                $stmt = $conn->prepare("UPDATE stocks SET name = ?, symbol = ?, minimum_amount = ?, maximum_amount = ?, cycle = ?, return_rate = ?, status = ? WHERE id = ?");
                $stmt->execute([$name, $symbol, $minimum, $maximum, $cycle, $returnRate, $status, $id]);
                $response['message'] = "Stock updated successfully.";
            } else {
                $stmt = $conn->prepare("INSERT INTO stocks (name, symbol, minimum_amount, maximum_amount, cycle, return_rate, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $symbol, $minimum, $maximum, $cycle, $returnRate, $status]);
                $response['message'] = "Stock created successfully.";
            }
            
            $response['success'] = true;
        } elseif ($_POST['action'] === 'delete' && isset($_POST['id'])) {
            $id = is_numeric($_POST['id']) ? $_POST['id'] : 0;
            $stmt = $conn->prepare("DELETE FROM stocks WHERE id = ?");
            $stmt->execute([$id]);
            $response['success'] = true;
            $response['message'] = "Stock deleted successfully.";
        } elseif ($_POST['action'] === 'get' && isset($_POST['id'])) {
            $id = is_numeric($_POST['id']) ? $_POST['id'] : 0;
            $stmt = $conn->prepare("SELECT * FROM stocks WHERE id = ?");
            $stmt->execute([$id]);
            $stock = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($stock) {
                $response['success'] = true;
                $response['data'] = $stock;
            } else {
                $response['message'] = "Stock not found.";
            }
        }
    } catch (Exception $e) {
        $response['message'] = $e->getMessage();
    }
    
    echo json_encode($response);
    exit;
}

// Get all stocks
try {
    $stocks = $conn->query("SELECT * FROM stocks ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Fetch stocks error: " . $e->getMessage());
    $stocks = [];
    $message = "Error fetching stocks.";
    $messageType = "error";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        :root {
            --sidebar-bg: #14213d;
            --sidebar-text: #a8b2d1;
            --sidebar-hover: #1a2c5b;
            --active-link: #2a4494;
            --primary-text: #333;
            --secondary-text: #777;
            --card-bg: #fff;
            --main-bg: #f0f2f5;
            --deposit-color: #3498db;
            --withdraw-color: #e74c3c;
            --button-color: #0d6efd;
            --hover-color: #0b5ed7;
        }

        body {
            display: flex;
            background-color: var(--main-bg);
            min-height: 100vh;
        }

        /* Mobile nav toggle */
        .mobile-nav-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            width: 40px;
            height: 40px;
            border-radius: 4px;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            border: none;
            font-size: 20px;
        }

        /* Overlay for mobile */
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 900;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            height: 100vh;
            position: fixed;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        .sidebar-header {
            padding: 20px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-menu {
            margin-top: 20px;
        }

        .section-title {
            padding: 10px 20px;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #8a94a7;
            margin-top: 10px;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            text-decoration: none;
            color: var(--sidebar-text);
            transition: all 0.3s ease;
        }

        .menu-item:hover {
            background-color: var(--sidebar-hover);
        }

        .menu-item.active {
            background-color: var(--active-link);
            color: white;
        }

        .menu-item i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 20px;
            width: calc(100% - 250px);
            transition: all 0.3s ease;
        }

        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .page-icon {
            font-size: 20px;
            color: var(--primary-text);
        }

        .user-info {
            display: flex;
            align-items: center;
        }

        .user-icon {
            width: 32px;
            height: 32px;
            background-color: var(--button-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-right: 10px;
        }

        .user-role {
            display: flex;
            flex-direction: column;
        }

        .role-name {
            font-size: 12px;
            color: var(--secondary-text);
        }

        /* Overview Section */
        .overview-section {
            margin-bottom: 30px;
        }

        .overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .overview-title {
            font-size: 24px;
            font-weight: 500;
            color: var(--primary-text);
        }

        .overview-description {
            font-size: 14px;
            color: var(--secondary-text);
            margin-top: 5px;
        }

        .view-users-btn {
            padding: 8px 16px;
            background-color: var(--button-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .view-users-btn:hover {
            background-color: var(--hover-color);
        }

        .view-users-btn i {
            margin-right: 6px;
        }

        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background-color: var(--card-bg);
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 20px;
            position: relative;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .card-title {
            font-size: 16px;
            color: var(--primary-text);
        }

        .card-icon {
            font-size: 14px;
            color: var(--secondary-text);
            cursor: pointer;
        }

        .card-amount {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .deposit-bar {
            height: 4px;
            width: 100%;
            background-color: var(--deposit-color);
            border-radius: 2px;
        }

        .withdraw-bar {
            height: 4px;
            width: 100%;
            background-color: var(--withdraw-color);
            border-radius: 2px;
        }

        /* Stats Card */
        .stats-card {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }

        .stat-item {
            margin-bottom: 15px;
        }

        .stat-label {
            display: flex;
            align-items: center;
            margin-bottom: 5px;
            font-size: 14px;
            color: var(--secondary-text);
        }

        .stat-label-circle {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .deposit-circle {
            background-color: #f39c12;
        }

        .withdraw-circle {
            background-color: #e74c3c;
        }

        .transaction-circle {
            background-color: #9b59b6;
        }

        .users-circle {
            background-color: #3498db;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 500;
        }

        /* Footer */
        .footer {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 40px;
            padding-top: 10px;
            border-top: 1px solid #eee;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .sidebar {
                width: 80px;
            }
            .sidebar-header {
                padding: 15px 10px;
                text-align: center;
                font-size: 14px;
            }
            .menu-item span, .section-title {
                display: none;
            }
            .menu-item {
                justify-content: center;
                padding: 15px 0;
            }
            .menu-item i {
                margin-right: 0;
                font-size: 18px;
            }
            .main-content {
                margin-left: 80px;
                width: calc(100% - 80px);
            }
        }

        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }
            .stats-card {
                grid-template-columns: 1fr;
            }
            .mobile-nav-toggle {
                display: flex;
            }
            .sidebar {
                transform: translateX(-100%);
                width: 250px;
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .menu-item span, .section-title {
                display: block;
            }
            .menu-item {
                justify-content: flex-start;
                padding: 12px 20px;
            }
            .menu-item i {
                margin-right: 10px;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding-top: 60px;
            }
            .overlay.active {
                display: block;
            }
        }

        @media (max-width: 576px) {
            .overview-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .view-users-btn {
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Mobile menu toggle -->
    <button class="mobile-nav-toggle" id="navToggle">
        <i class="fas fa-bars"></i>
    </button>

             <!-- Overlay for closing the sidebar on mobile -->
    <div class="overlay" id="overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header">ADMIN DASHBOARD</div>
        <div class="sidebar-menu">
            <a href="/admin/admin_dashboard.php" class="menu-item active">
                <i class="fas fa-th-large"></i>
                <span>Dashboard</span>
            </a>
            <a href="/admin/coin-deposit-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-down"></i>
                <span>Deposit Coin</span>
            </a>
             <a href="/admin/deposit_requests.php" class="menu-item">
                <i class="fas fa-money-check-alt"></i>
                <span>Deposit Check</span>
            </a>
            <a href="/admin/transfer-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-up"></i>
                <span>Withdraw</span>
            </a>
            <a href="/admin/manage-payment-method.php" class="menu-item">
                <i class="fas fa-exchange-alt"></i>
                <span>Manage Payment Method</span>
            </a>
            <a href="/admin/admin_staking.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking</span>
            </a>
            <a href="/admin/admin_staking_transactions.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking Transations</span>
            </a>
            <a href="/admin/admin_stocks.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks</span>
            </a>
            <a href="/admin/admin_stock_transactions.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks Transations</span>
            </a>
            <a href="/admin/admin_bots.php" class="menu-item">
                <i class="fas fa-robot"></i>
                <span>Manage Bot</span>
            </a>
            <a href="/admin/investment.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Investment</span>
            </a>
            
            <a href="/admin/crypto_orders.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Crypto Order</span>
            </a>
            
            <a href="/admin/update_balance.php" class="menu-item">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Update Balances</span>
            </a>
            <a href="/admin/kyc-request.php" class="menu-item">
                <i class="fas fa-id-card"></i>
                <span>KYC</span>
            </a>
            <a href="/admin/admin_send_notification.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Manage Notifications</span>
            </a>

            <div class="section-title">USER MANAGEMENT</div>
            <a href="/admin/manage_users.php" class="menu-item">
                <i class="fas fa-users"></i>
                <span>Manage Users</span>
            </a>
            <div class="section-title">Approve SignUp</div>
            <a href="/admin/admin_approval.php" class="menu-item">
                <i class="fas fa-user-check"></i>
                <span>Approve SignUp</span>
            </a>
            <div class="section-title">Change Password</div>
            <a href="/admin/change_password.php" class="menu-item">
                <i class="fas fa-unlock-alt"></i>
                <span>Change Password</span>
            </a>

            <div class="section-title">MANAGEMENT</div>
            <a href="/admin/logout.php" class="menu-item">
                <i class="fas fa-sign-out"></i>
                <span>Sign Out</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="top-bar">
            <div class="page-icon">
                <i class="fas fa-clipboard"></i>
            </div>
            <div class="user-info">
                <div class="user-icon">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-role">
                    <div>Admin</div>
                    <div class="role-name">Administrator <i class="fas fa-chevron-down"></i></div>
                </div>
            </div>
        </div>

        <!-- Overview Section -->
        <div class="overview-section">
            <div class="overview-header">
                <div>
                    <h2 class="overview-title">Overview</h2>
                    <p class="overview-description">Here is an insight of what's going on.</p>
                </div>
                <button class="view-users-btn">
                    <i class="fas fa-users"></i> View All Users
                </button>
            </div>

    <style>
        h1 { color: #333; }
        
        /* Table Styles */
        table { width: 100%; border-collapse: collapse; margin-top: 20px; background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
        table, th, td { border: 1px solid #ddd; }
        th, td { padding: 12px; text-align: left; }
        th { background-color: #f8f8f8; }
        tr:nth-child(even) { background-color: #f9f9f9; }
        tr:hover { background-color: #f1f1f1; }
        
        /* Button Styles */
        .btn { padding: 8px 16px; background: #006b91; color: white; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; display: inline-block; margin-right: 5px; }
        .btn:hover { background: #004f6b; }
        .delete-btn { background: #d9534f; }
        .delete-btn:hover { background: #c9302c; }
        .add-btn { margin-bottom: 20px; background: #5cb85c; }
        .add-btn:hover { background: #4cae4c; }
        
        /* Form Styles */
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; }
        .form-group input, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        
        /* Modal Styles */
        .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4); }
        .modal-content { background-color: #fefefe; margin: 10% auto; padding: 20px; border: 1px solid #888; width: 50%; box-shadow: 0 4px 8px rgba(0,0,0,0.1); border-radius: 5px; }
        .close { color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer; }
        .close:hover { color: black; }
        
        /* Message Styles */
        #message-container { padding: 10px; margin: 20px 0; border-radius: 4px; display: none; }
        .success { background-color: #dff0d8; color: #3c763d; border: 1px solid #d6e9c6; }
        .error { background-color: #f2dede; color: #a94442; border: 1px solid #ebccd1; }

        /* Loading overlay */
        .loading-overlay { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 2000; justify-content: center; align-items: center; }
        .loading-spinner { border: 5px solid #f3f3f3; border-top: 5px solid #3498db; border-radius: 50%; width: 50px; height: 50px; animation: spin 1s linear infinite; }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    </style>
    <h1>Admin Panel: Manage Stock Plans</h1>

<div id="message-container"></div>

<button id="add-stock-btn" class="btn add-btn">Add New Stock Plan</button>

<table id="stocks-table">
    <thead>
        <tr>
            <th>ID</th>
            <th>Name</th>
            <th>Symbol</th>
            <th>Minimum</th>
            <th>Maximum</th>
            <th>Cycle</th>
            <th>Return Rate</th>
            <th>Status</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php if (empty($stocks)): ?>
            <tr>
                <td colspan="9" style="text-align: center;">No stocks found</td>
            </tr>
        <?php else: ?>
            <?php foreach ($stocks as $stock): ?>
            <tr data-id="<?= $stock['id'] ?>">
                <td><?= htmlspecialchars($stock['id']) ?></td>
                <td><?= htmlspecialchars($stock['name']) ?></td>
                <td><?= htmlspecialchars($stock['symbol']) ?></td>
                <td><?= number_format($stock['minimum_amount'], 2) ?> USDT</td>
                <td><?= number_format($stock['maximum_amount'], 2) ?> USDT</td>
                <td><?= htmlspecialchars($stock['cycle']) ?></td>
                <td><?= number_format($stock['return_rate'], 2) ?>%</td>
                <td><?= htmlspecialchars($stock['status']) ?></td>
                <td>
                    <button class="btn edit-btn" data-id="<?= $stock['id'] ?>">Edit</button>
                    <button class="btn delete-btn" data-id="<?= $stock['id'] ?>">Delete</button>
                </td>
            </tr>
            <?php endforeach; ?>
        <?php endif; ?>
    </tbody>
</table>

    <!-- Modal for Create/Edit -->
    <div id="stock-modal" class="modal">
    <div class="modal-content">
        <span class="close">&times;</span>
        <h2 id="modal-title">Create Stock Plan</h2>
        <form id="stock-form">
            <input type="hidden" id="stock-id" name="id" value="">
            <input type="hidden" id="action" name="action" value="create">
            
            <div class="form-group">
                <label for="name">Stock Name:</label>
                <input type="text" id="name" name="name" required placeholder="Stock Name">
            </div>
            
            <div class="form-group">
                <label for="symbol">Symbol:</label>
                <input type="text" id="symbol" name="symbol" required placeholder="Symbol (e.g. BTC)">
            </div>
            
            <div class="form-group">
                <label for="minimum">Minimum Amount (USDT):</label>
                <input type="number" id="minimum" step="0.01" min="0" name="minimum" required placeholder="Minimum Amount">
            </div>
            
            <div class="form-group">
                <label for="maximum">Maximum Amount (USDT):</label>
                <input type="number" id="maximum" step="0.01" min="0" name="maximum" required placeholder="Maximum Amount">
            </div>
            
            <div class="form-group">
                <label for="cycle">Cycle:</label>
                <input type="text" id="cycle" name="cycle" required placeholder="Cycle (e.g. 7 days)">
            </div>
            
            <div class="form-group">
                <label for="return_rate">Return Rate (%):</label>
                <input type="number" id="return_rate" step="0.01" name="return_rate" required 
                       placeholder="Expected return rate">
            </div>
            
            <div class="form-group">
                <label for="status">Status:</label>
                <select id="status" name="status" required>
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                </select>
            </div>
            
            <button type="submit" class="btn" id="submit-btn">Create Stock Plan</button>
        </form>
    </div>
</div>

    <!-- Delete Confirmation Modal -->
    <div id="delete-modal" class="modal">
        <div class="modal-content">
            <span class="close">&times;</span>
            <h2>Confirm Delete</h2>
            <p>Are you sure you want to delete this stock plan?</p>
            <div>
                <button id="confirm-delete" class="btn delete-btn">Delete</button>
                <button id="cancel-delete" class="btn">Cancel</button>
            </div>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loading-overlay">
        <div class="loading-spinner"></div>
    </div>

    <script>
     // Get references to elements
        const navToggle = document.getElementById('navToggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');
        const mainContent = document.getElementById('mainContent');

        // Toggle sidebar
        navToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        });

        // Close sidebar when clicking on overlay
        overlay.addEventListener('click', function() {
            sidebar.classList.remove('active');
            overlay.classList.remove('active');
        });

        // Close sidebar when clicking on main content (for mobile)
        mainContent.addEventListener('click', function() {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Close sidebar when clicking menu items (for mobile)
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                }
            });
        });
        document.addEventListener('DOMContentLoaded', function() {
            // Elements
            const stockModal = document.getElementById('stock-modal');
            const deleteModal = document.getElementById('delete-modal');
            const modalTitle = document.getElementById('modal-title');
            const stockForm = document.getElementById('stock-form');
            const addStockBtn = document.getElementById('add-stock-btn');
            const submitBtn = document.getElementById('submit-btn');
            const stocksTable = document.getElementById('stocks-table');
            const closeButtons = document.querySelectorAll('.close');
            const loadingOverlay = document.getElementById('loading-overlay');
            const messageContainer = document.getElementById('message-container');
            let currentDeleteId = null;

            // Show loading
            function showLoading() {
                loadingOverlay.style.display = 'flex';
            }

            // Hide loading
            function hideLoading() {
                loadingOverlay.style.display = 'none';
            }

            // Show message
            function showMessage(message, type) {
                messageContainer.textContent = message;
                messageContainer.className = type;
                messageContainer.style.display = 'block';
                setTimeout(() => {
                    messageContainer.style.display = 'none';
                }, 5000);
            }

            // Reset form
            function resetForm() {
                stockForm.reset();
                document.getElementById('stock-id').value = '';
                document.getElementById('action').value = 'create';
                modalTitle.textContent = 'Create Stock Plan';
                submitBtn.textContent = 'Create Stock Plan';
            }

            // Open modal for adding new stock
            addStockBtn.addEventListener('click', function() {
                resetForm();
                stockModal.style.display = 'block';
            });

            // Handle edit button click
            stocksTable.addEventListener('click', function(e) {
                if (e.target.classList.contains('edit-btn')) {
                    const id = e.target.getAttribute('data-id');
                    showLoading();
                    
                    // Fetch stock data
                    fetch('admin_stocks.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            'action': 'get',
                            'id': id
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        hideLoading();
                        if (data.success) {
                            document.getElementById('stock-id').value = data.data.id;
                            document.getElementById('action').value = 'update';
                            document.getElementById('name').value = data.data.name;
                            document.getElementById('symbol').value = data.data.symbol;
                            document.getElementById('minimum').value = data.data.minimum_amount;
                            document.getElementById('maximum').value = data.data.maximum_amount;
                            document.getElementById('cycle').value = data.data.cycle;
                            document.getElementById('status').value = data.data.status;
                            
                            modalTitle.textContent = 'Edit Stock Plan';
                            submitBtn.textContent = 'Update Stock Plan';
                            stockModal.style.display = 'block';
                        } else {
                            showMessage(data.message || 'Error fetching stock data', 'error');
                        }
                    })
                    .catch(error => {
                        hideLoading();
                        showMessage('Error: ' + error.message, 'error');
                    });
                } else if (e.target.classList.contains('delete-btn')) {
                    const id = e.target.getAttribute('data-id');
                    currentDeleteId = id;
                    deleteModal.style.display = 'block';
                }
            });

            // Handle form submission
            stockForm.addEventListener('submit', function(e) {
                e.preventDefault();
                showLoading();
                
                const formData = new FormData(stockForm);
                
                fetch('admin_stocks.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        stockModal.style.display = 'none';
                        showMessage(data.message, 'success');
                        // Reload the page to see changes
                        window.location.reload();
                    } else {
                        showMessage(data.message || 'Error processing request', 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showMessage('Error: ' + error.message, 'error');
                });
            });

            // Handle delete confirmation
            document.getElementById('confirm-delete').addEventListener('click', function() {
                if (currentDeleteId) {
                    showLoading();
                    
                    fetch('admin_stocks.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            'action': 'delete',
                            'id': currentDeleteId
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        hideLoading();
                        deleteModal.style.display = 'none';
                        if (data.success) {
                            showMessage(data.message, 'success');
                            // Remove the row from table
                            const row = document.querySelector(`tr[data-id="${currentDeleteId}"]`);
                            if (row) row.remove();
                            
                            // If no more rows, add a "No stocks found" message
                            const tbody = stocksTable.querySelector('tbody');
                            if (tbody.children.length === 0) {
                                const noDataRow = document.createElement('tr');
                                noDataRow.innerHTML = '<td colspan="8" style="text-align: center;">No stocks found</td>';
                                tbody.appendChild(noDataRow);
                            }
                        } else {
                            showMessage(data.message || 'Error deleting stock', 'error');
                        }
                    })
                    .catch(error => {
                        hideLoading();
                        deleteModal.style.display = 'none';
                        showMessage('Error: ' + error.message, 'error');
                    });
                }
            });

            // Handle cancel delete
            document.getElementById('cancel-delete').addEventListener('click', function() {
                deleteModal.style.display = 'none';
                currentDeleteId = null;
            });

            // Close modals when clicking the X
            closeButtons.forEach(function(closeBtn) {
                closeBtn.addEventListener('click', function() {
                    stockModal.style.display = 'none';
                    deleteModal.style.display = 'none';
                    currentDeleteId = null;
                });
            });

            // Close modals when clicking outside
            window.addEventListener('click', function(e) {
                if (e.target === stockModal) {
                    stockModal.style.display = 'none';
                }
                if (e.target === deleteModal) {
                    deleteModal.style.display = 'none';
                    currentDeleteId = null;
                }
            });
        });
        // In your modal handling code, make sure to include return_rate
function loadStockData(id) {
    $.post('admin_stocks.php', {action: 'get', id: id}, function(response) {
        if (response.success) {
            $('#stock-id').val(response.data.id);
            $('#name').val(response.data.name);
            $('#symbol').val(response.data.symbol);
            $('#minimum').val(response.data.minimum_amount);
            $('#maximum').val(response.data.maximum_amount);
            $('#cycle').val(response.data.cycle);
            $('#return_rate').val(response.data.return_rate); // Add this line
            $('#status').val(response.data.status);
            $('#action').val('update');
            $('#modal-title').text('Edit Stock Plan');
            $('#submit-btn').text('Update Stock Plan');
            $('#stock-modal').show();
        }
    }, 'json');
}
    </script>
</body>
</html>