<?php
// Secure configuration and initialization
require_once($_SERVER['DOCUMENT_ROOT'] . '/config.php');
session_start();
// Verify admin is logged in
if (!isset($_SESSION['admin_id']) || empty($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}
// Error logging setup
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/error.log');

// Initialize variables
$error = '';
$success = '';
$plans = [];
$editPlan = null;

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Determine action type
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'create':
                    createPlan($conn);
                    break;
                    
                case 'update':
                    updatePlan($conn);
                    break;
                    
                case 'delete':
                    deletePlan($conn);
                    break;
                    
                default:
                    throw new Exception("Unknown action");
            }
        } else {
            throw new Exception("Missing action parameter");
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Handle GET requests for editing
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    try {
        $editId = (int)$_GET['edit'];
        $editStmt = $conn->prepare("SELECT * FROM staking_plans WHERE id = :id");
        $editStmt->execute(['id' => $editId]);
        $editPlan = $editStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$editPlan) {
            $error = "Plan not found.";
        }
    } catch (Exception $e) {
        $error = "Error retrieving plan: " . $e->getMessage();
    }
}

// Fetch all staking plans
try {
    $plansStmt = $conn->prepare("SELECT * FROM staking_plans ORDER BY id DESC");
    $plansStmt->execute();
    $plans = $plansStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Error fetching plans: " . $e->getMessage());
    $error = "Unable to load staking plans.";
}

/**
 * Create a new staking plan
 * @param PDO $conn Database connection
 */
function createPlan($conn) {
    global $success, $error;
    
    // Validate inputs
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_STRING);
    $symbol = filter_input(INPUT_POST, 'symbol', FILTER_SANITIZE_STRING);
    $cycle = filter_input(INPUT_POST, 'cycle', FILTER_SANITIZE_STRING);
    $minAmount = filter_input(INPUT_POST, 'minimum_amount', FILTER_VALIDATE_FLOAT);
    $maxAmount = filter_input(INPUT_POST, 'maximum_amount', FILTER_VALIDATE_FLOAT);
    $returnRate = filter_input(INPUT_POST, 'return_rate', FILTER_VALIDATE_FLOAT);
    $status = filter_input(INPUT_POST, 'status', FILTER_SANITIZE_STRING);
    
    if (!$name || !$symbol || !$cycle || $minAmount === false || $maxAmount === false || $returnRate === false || !$status) {
        throw new Exception("All fields are required with valid values.");
    }
    
    if ($minAmount <= 0) {
        throw new Exception("Minimum amount must be greater than zero.");
    }
    
    if ($maxAmount <= $minAmount) {
        throw new Exception("Maximum amount must be greater than minimum amount.");
    }
    
    if ($returnRate < -100 || $returnRate > 1000) {
        throw new Exception("Return rate must be between -100% and 1000%");
    }
    
    // Insert new plan
    $insertStmt = $conn->prepare("
        INSERT INTO staking_plans 
        (name, symbol, cycle, minimum_amount, maximum_amount, return_rate, status, created_at) 
        VALUES (:name, :symbol, :cycle, :min_amount, :max_amount, :return_rate, :status, NOW())
    ");
    
    $insertStmt->execute([
        'name' => $name,
        'symbol' => $symbol,
        'cycle' => $cycle,
        'min_amount' => $minAmount,
        'max_amount' => $maxAmount,
        'return_rate' => $returnRate,
        'status' => $status
    ]);
    
    $success = "Staking plan created successfully.";
}

/**
 * Update an existing staking plan
 * @param PDO $conn Database connection
 */
function updatePlan($conn) {
    global $success, $error;
    
    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_STRING);
    $symbol = filter_input(INPUT_POST, 'symbol', FILTER_SANITIZE_STRING);
    $cycle = filter_input(INPUT_POST, 'cycle', FILTER_SANITIZE_STRING);
    $minAmount = filter_input(INPUT_POST, 'minimum_amount', FILTER_VALIDATE_FLOAT);
    $maxAmount = filter_input(INPUT_POST, 'maximum_amount', FILTER_VALIDATE_FLOAT);
    $returnRate = filter_input(INPUT_POST, 'return_rate', FILTER_VALIDATE_FLOAT);
    $status = filter_input(INPUT_POST, 'status', FILTER_SANITIZE_STRING);
    
    if (!$id || !$name || !$symbol || !$cycle || $minAmount === false || $maxAmount === false || $returnRate === false || !$status) {
        throw new Exception("All fields are required with valid values.");
    }
    
    if ($minAmount <= 0) {
        throw new Exception("Minimum amount must be greater than zero.");
    }
    
    if ($maxAmount <= $minAmount) {
        throw new Exception("Maximum amount must be greater than minimum amount.");
    }
    
    if ($returnRate < -100 || $returnRate > 1000) {
        throw new Exception("Return rate must be between -100% and 1000%");
    }
    
    // First, check if plan exists
    $checkStmt = $conn->prepare("SELECT id FROM staking_plans WHERE id = :id");
    $checkStmt->execute(['id' => $id]);
    
    if (!$checkStmt->fetch()) {
        throw new Exception("Plan not found.");
    }
    
    // Update plan
    $updateStmt = $conn->prepare("
        UPDATE staking_plans 
        SET name = :name, 
            symbol = :symbol, 
            cycle = :cycle, 
            minimum_amount = :min_amount, 
            maximum_amount = :max_amount,
            return_rate = :return_rate,
            status = :status, 
            updated_at = NOW() 
        WHERE id = :id
    ");
    
    $updateStmt->execute([
        'id' => $id,
        'name' => $name,
        'symbol' => $symbol,
        'cycle' => $cycle,
        'min_amount' => $minAmount,
        'max_amount' => $maxAmount,
        'return_rate' => $returnRate,
        'status' => $status
    ]);
    
    $success = "Staking plan updated successfully.";
}

/**
 * Delete a staking plan
 * @param PDO $conn Database connection
 */
function deletePlan($conn) {
    global $success, $error;
    
    $id = filter_input(INPUT_POST, 'delete_id', FILTER_VALIDATE_INT);
    
    if (!$id) {
        throw new Exception("Invalid plan ID.");
    }
    
    // Check if plan exists
    $checkStmt = $conn->prepare("SELECT id FROM staking_plans WHERE id = :id");
    $checkStmt->execute(['id' => $id]);
    
    if (!$checkStmt->fetch()) {
        throw new Exception("Plan not found.");
    }
    
    // Check if plan is in use
    $useCheckStmt = $conn->prepare("SELECT COUNT(*) as count FROM staking_history WHERE plan_id = :plan_id");
    $useCheckStmt->execute(['plan_id' => $id]);
    $result = $useCheckStmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result['count'] > 0) {
        // Set to inactive instead of deleting
        $updateStmt = $conn->prepare("UPDATE staking_plans SET status = 'inactive', updated_at = NOW() WHERE id = :id");
        $updateStmt->execute(['id' => $id]);
        $success = "Plan has active stakes and cannot be deleted. Status changed to inactive.";
    } else {
        // Delete the plan
        $deleteStmt = $conn->prepare("DELETE FROM staking_plans WHERE id = :id");
        $deleteStmt->execute(['id' => $id]);
        $success = "Staking plan deleted successfully.";
    }
}

// Close connection
$conn = null;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        :root {
            --sidebar-bg: #14213d;
            --sidebar-text: #a8b2d1;
            --sidebar-hover: #1a2c5b;
            --active-link: #2a4494;
            --primary-text: #333;
            --secondary-text: #777;
            --card-bg: #fff;
            --main-bg: #f0f2f5;
            --deposit-color: #3498db;
            --withdraw-color: #e74c3c;
            --button-color: #0d6efd;
            --hover-color: #0b5ed7;
        }

        body {
            display: flex;
            background-color: var(--main-bg);
            min-height: 100vh;
        }

        /* Mobile nav toggle */
        .mobile-nav-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            width: 40px;
            height: 40px;
            border-radius: 4px;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            border: none;
            font-size: 20px;
        }

        /* Overlay for mobile */
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 900;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            height: 100vh;
            position: fixed;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        .sidebar-header {
            padding: 20px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-menu {
            margin-top: 20px;
        }

        .section-title {
            padding: 10px 20px;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #8a94a7;
            margin-top: 10px;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            text-decoration: none;
            color: var(--sidebar-text);
            transition: all 0.3s ease;
        }

        .menu-item:hover {
            background-color: var(--sidebar-hover);
        }

        .menu-item.active {
            background-color: var(--active-link);
            color: white;
        }

        .menu-item i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 20px;
            width: calc(100% - 250px);
            transition: all 0.3s ease;
        }

        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .page-icon {
            font-size: 20px;
            color: var(--primary-text);
        }

        .user-info {
            display: flex;
            align-items: center;
        }

        .user-icon {
            width: 32px;
            height: 32px;
            background-color: var(--button-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-right: 10px;
        }

        .user-role {
            display: flex;
            flex-direction: column;
        }

        .role-name {
            font-size: 12px;
            color: var(--secondary-text);
        }

        /* Overview Section */
        .overview-section {
            margin-bottom: 30px;
        }

        .overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .overview-title {
            font-size: 24px;
            font-weight: 500;
            color: var(--primary-text);
        }

        .overview-description {
            font-size: 14px;
            color: var(--secondary-text);
            margin-top: 5px;
        }

        .view-users-btn {
            padding: 8px 16px;
            background-color: var(--button-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .view-users-btn:hover {
            background-color: var(--hover-color);
        }

        .view-users-btn i {
            margin-right: 6px;
        }

        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background-color: var(--card-bg);
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 20px;
            position: relative;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .card-title {
            font-size: 16px;
            color: var(--primary-text);
        }

        .card-icon {
            font-size: 14px;
            color: var(--secondary-text);
            cursor: pointer;
        }

        .card-amount {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .deposit-bar {
            height: 4px;
            width: 100%;
            background-color: var(--deposit-color);
            border-radius: 2px;
        }

        .withdraw-bar {
            height: 4px;
            width: 100%;
            background-color: var(--withdraw-color);
            border-radius: 2px;
        }

        /* Stats Card */
        .stats-card {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }

        .stat-item {
            margin-bottom: 15px;
        }

        .stat-label {
            display: flex;
            align-items: center;
            margin-bottom: 5px;
            font-size: 14px;
            color: var(--secondary-text);
        }

        .stat-label-circle {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .deposit-circle {
            background-color: #f39c12;
        }

        .withdraw-circle {
            background-color: #e74c3c;
        }

        .transaction-circle {
            background-color: #9b59b6;
        }

        .users-circle {
            background-color: #3498db;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 500;
        }

        /* Footer */
        .footer {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 40px;
            padding-top: 10px;
            border-top: 1px solid #eee;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .sidebar {
                width: 80px;
            }
            .sidebar-header {
                padding: 15px 10px;
                text-align: center;
                font-size: 14px;
            }
            .menu-item span, .section-title {
                display: none;
            }
            .menu-item {
                justify-content: center;
                padding: 15px 0;
            }
            .menu-item i {
                margin-right: 0;
                font-size: 18px;
            }
            .main-content {
                margin-left: 80px;
                width: calc(100% - 80px);
            }
        }

        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }
            .stats-card {
                grid-template-columns: 1fr;
            }
            .mobile-nav-toggle {
                display: flex;
            }
            .sidebar {
                transform: translateX(-100%);
                width: 250px;
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .menu-item span, .section-title {
                display: block;
            }
            .menu-item {
                justify-content: flex-start;
                padding: 12px 20px;
            }
            .menu-item i {
                margin-right: 10px;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding-top: 60px;
            }
            .overlay.active {
                display: block;
            }
        }

        @media (max-width: 576px) {
            .overview-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .view-users-btn {
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Mobile menu toggle -->
    <button class="mobile-nav-toggle" id="navToggle">
        <i class="fas fa-bars"></i>
    </button>

            <!-- Overlay for closing the sidebar on mobile -->
    <div class="overlay" id="overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header">ADMIN DASHBOARD</div>
        <div class="sidebar-menu">
            <a href="/admin/admin_dashboard.php" class="menu-item active">
                <i class="fas fa-th-large"></i>
                <span>Dashboard</span>
            </a>
            <a href="/admin/coin-deposit-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-down"></i>
                <span>Deposit Coin</span>
            </a>
             <a href="/admin/deposit_requests.php" class="menu-item">
                <i class="fas fa-money-check-alt"></i>
                <span>Deposit Check</span>
            </a>
            <a href="/admin/transfer-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-up"></i>
                <span>Withdraw</span>
            </a>
            <a href="/admin/manage-payment-method.php" class="menu-item">
                <i class="fas fa-exchange-alt"></i>
                <span>Manage Payment Method</span>
            </a>
            <a href="/admin/admin_staking.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking</span>
            </a>
            <a href="/admin/admin_staking_transactions.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking Transations</span>
            </a>
            <a href="/admin/admin_stocks.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks</span>
            </a>
            <a href="/admin/admin_stock_transactions.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks Transations</span>
            </a>
            <a href="/admin/admin_bots.php" class="menu-item">
                <i class="fas fa-robot"></i>
                <span>Manage Bot</span>
            </a>
            <a href="/admin/investment.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Investment</span>
            </a>
            
            <a href="/admin/crypto_orders.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Crypto Order</span>
            </a>
            
            <a href="/admin/update_balance.php" class="menu-item">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Update Balances</span>
            </a>
            <a href="/admin/kyc-request.php" class="menu-item">
                <i class="fas fa-id-card"></i>
                <span>KYC</span>
            </a>
            <a href="/admin/admin_send_notification.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Manage Notifications</span>
            </a>

            <div class="section-title">USER MANAGEMENT</div>
            <a href="/admin/manage_users.php" class="menu-item">
                <i class="fas fa-users"></i>
                <span>Manage Users</span>
            </a>
            <div class="section-title">Approve SignUp</div>
            <a href="/admin/admin_approval.php" class="menu-item">
                <i class="fas fa-user-check"></i>
                <span>Approve SignUp</span>
            </a>
            <div class="section-title">Change Password</div>
            <a href="/admin/change_password.php" class="menu-item">
                <i class="fas fa-unlock-alt"></i>
                <span>Change Password</span>
            </a>

            <div class="section-title">MANAGEMENT</div>
            <a href="/admin/logout.php" class="menu-item">
                <i class="fas fa-sign-out"></i>
                <span>Sign Out</span>
            </a>
        </div>
    </div>


    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="top-bar">
            <div class="page-icon">
                <i class="fas fa-clipboard"></i>
            </div>
            <div class="user-info">
                <div class="user-icon">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-role">
                    <div>Admin</div>
                    <div class="role-name">Administrator <i class="fas fa-chevron-down"></i></div>
                </div>
            </div>
        </div>

        <!-- Overview Section -->
        <div class="overview-section">
            <div class="overview-header">
                <div>
                    <h2 class="overview-title">Overview</h2>
                    <p class="overview-description">Here is an insight of what's going on.</p>
                </div>
                <button class="view-users-btn">
                    <i class="fas fa-users"></i> View All Users
                </button>
            </div>

    <style>
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .alert {
            padding: 12px;
            margin-bottom: 20px;
            border-radius: 4px;
        }
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .admin-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        .admin-header h2 {
            margin: 0;
        }
        .btn {
            display: inline-block;
            padding: 8px 16px;
            background-color: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        .btn-primary {
            background-color: #007bff;
        }
        .btn-success {
            background-color: #28a745;
        }
        .btn-danger {
            background-color: #dc3545;
        }
        .btn-warning {
            background-color: #ffc107;
            color: #212529;
        }
        .btn:hover {
            opacity: 0.9;
        }
        .plans-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
        }
        .plans-table th, .plans-table td {
            padding: 12px;
            text-align: left;
            border: 1px solid #ddd;
        }
        .plans-table th {
            background-color: #f2f2f2;
            font-weight: bold;
        }
        .plans-table tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .form-section {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 30px;
        }
        .form-row {
            margin-bottom: 15px;
        }
        .form-row label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .form-row input, .form-row select {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .form-actions {
            margin-top: 20px;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        .status-active {
            color: #28a745;
            font-weight: bold;
        }
        .status-inactive {
            color: #dc3545;
            font-weight: bold;
        }
        .actions-cell {
            white-space: nowrap;
        }
    </style>
    <div class="container">
    <div class="admin-header">
        <h2>Staking Plans Administration</h2>
        <div>
            <a href="admin_staking.php" class="btn">Back to Staking</a>
            <a href="admin_dashboard.php" class="btn">Dashboard</a>
        </div>
    </div>
    
    <?php if (!empty($error)): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>
    
    <?php if (!empty($success)): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>
    
    <!-- Create/Edit Form -->
    <div class="form-section">
        <h3><?= $editPlan ? 'Edit Staking Plan' : 'Create New Staking Plan' ?></h3>
        <form method="post">
            <input type="hidden" name="action" value="<?= $editPlan ? 'update' : 'create' ?>">
            <?php if ($editPlan): ?>
                <input type="hidden" name="id" value="<?= $editPlan['id'] ?>">
            <?php endif; ?>
            
            <div class="form-row">
                <label for="name">Plan Name:</label>
                <input type="text" id="name" name="name" value="<?= $editPlan ? htmlspecialchars($editPlan['name']) : '' ?>" required>
            </div>
            
            <div class="form-row">
                <label for="symbol">Symbol:</label>
                <input type="text" id="symbol" name="symbol" value="<?= $editPlan ? htmlspecialchars($editPlan['symbol']) : '' ?>" required>
            </div>
            
            <div class="form-row">
                <label for="cycle">Cycle:</label>
                <input type="text" id="cycle" name="cycle" value="<?= $editPlan ? htmlspecialchars($editPlan['cycle']) : '' ?>" placeholder="e.g., Daily, Weekly, Monthly" required>
            </div>
            
            <div class="form-row">
                <label for="minimum_amount">Minimum Amount:</label>
                <input type="number" id="minimum_amount" name="minimum_amount" step="0.01" min="0.01" value="<?= $editPlan ? $editPlan['minimum_amount'] : '' ?>" required>
            </div>
            
            <div class="form-row">
                <label for="maximum_amount">Maximum Amount:</label>
                <input type="number" id="maximum_amount" name="maximum_amount" step="0.01" min="0.01" value="<?= $editPlan ? $editPlan['maximum_amount'] : '' ?>" required>
            </div>
            
            <div class="form-row">
                <label for="return_rate">Return Rate (%):</label>
                <input type="number" id="return_rate" name="return_rate" step="0.01" value="<?= $editPlan ? $editPlan['return_rate'] : '' ?>" required>
            </div>
            
            <div class="form-row">
                <label for="status">Status:</label>
                <select id="status" name="status" required>
                    <option value="active" <?= ($editPlan && $editPlan['status'] == 'active') ? 'selected' : '' ?>>Active</option>
                    <option value="inactive" <?= ($editPlan && $editPlan['status'] == 'inactive') ? 'selected' : '' ?>>Inactive</option>
                </select>
            </div>
            
            <div class="form-actions">
                <?php if ($editPlan): ?>
                    <a href="admin_staking.php" class="btn btn-warning">Cancel</a>
                <?php endif; ?>
                <button type="submit" class="btn btn-success"><?= $editPlan ? 'Update Plan' : 'Create Plan' ?></button>
            </div>
        </form>
    </div>
    
    <!-- Plans Table -->
    <h3>Existing Staking Plans</h3>
    <?php if (empty($plans)): ?>
        <p>No staking plans found.</p>
    <?php else: ?>
        <table class="plans-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Symbol</th>
                    <th>Cycle</th>
                    <th>Min Amount</th>
                    <th>Max Amount</th>
                    <th>Return Rate</th>
                    <th>Status</th>
                    <th>Created</th>
                    <th>Updated</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($plans as $plan): ?>
                    <tr>
                        <td><?= $plan['id'] ?></td>
                        <td><?= htmlspecialchars($plan['name']) ?></td>
                        <td><?= htmlspecialchars($plan['symbol']) ?></td>
                        <td><?= htmlspecialchars($plan['cycle']) ?></td>
                        <td><?= number_format($plan['minimum_amount'], 2) ?></td>
                        <td><?= number_format($plan['maximum_amount'], 2) ?></td>
                        <td><?= number_format($plan['return_rate'], 2) ?>%</td>
                        <td class="status-<?= $plan['status'] ?>">
                            <?= strtoupper(htmlspecialchars($plan['status'])) ?>
                        </td>
                        <td><?= date('Y-m-d', strtotime($plan['created_at'])) ?></td>
                        <td><?= $plan['updated_at'] ? date('Y-m-d', strtotime($plan['updated_at'])) : 'N/A' ?></td>
                        <td class="actions-cell">
                            <a href="?edit=<?= $plan['id'] ?>" class="btn btn-warning">Edit</a>
                            <form method="post" style="display:inline;" onsubmit="return confirm('Are you sure you want to delete this plan?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="delete_id" value="<?= $plan['id'] ?>">
                                <button type="submit" class="btn btn-danger">Delete</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

    <script>
     // Get references to elements
        const navToggle = document.getElementById('navToggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');
        const mainContent = document.getElementById('mainContent');

        // Toggle sidebar
        navToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        });

        // Close sidebar when clicking on overlay
        overlay.addEventListener('click', function() {
            sidebar.classList.remove('active');
            overlay.classList.remove('active');
        });

        // Close sidebar when clicking on main content (for mobile)
        mainContent.addEventListener('click', function() {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Close sidebar when clicking menu items (for mobile)
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                }
            });
        });
        // Validation script
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form');
            
            form.addEventListener('submit', function(e) {
                const minAmount = parseFloat(document.getElementById('minimum_amount').value);
                const maxAmount = parseFloat(document.getElementById('maximum_amount').value);
                
                if (minAmount <= 0) {
                    e.preventDefault();
                    alert('Minimum amount must be greater than zero.');
                    return false;
                }
                
                if (maxAmount <= minAmount) {
                    e.preventDefault();
                    alert('Maximum amount must be greater than minimum amount.');
                    return false;
                }
                
                return true;
            });
        });
    </script>
</body>
</html>